/**
 * \file pappsomspp/core/msrun/private/mzcbormsrunreader.h
 * \date 21/11/2025
 * \author Olivier Langella
 * \brief MSrun file reader for mzcbor
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#pragma once

#include <QFileInfo>
#include "pappsomspp/core/msrun/msrunreader.h"
#include "pappsomspp/core/processing/cbor/mzcbor/spectrum.h"

namespace pappso
{

/**
 * @todo write docs
 */
class MzcborMsRunReader : public MsRunReader
{
  public:
  /**
   * Default constructor
   */
  MzcborMsRunReader(MsRunIdCstSPtr &msrun_id_csp);

  /**
   * Destructor
   */
  virtual ~MzcborMsRunReader();


  virtual MassSpectrumSPtr massSpectrumSPtr(std::size_t spectrum_index) override;
  virtual MassSpectrumCstSPtr massSpectrumCstSPtr(std::size_t spectrum_index) override;

  virtual QualifiedMassSpectrum qualifiedMassSpectrum(std::size_t spectrum_index,
                                                      bool want_binary_data = true) const override;

  virtual void readSpectrumCollection(SpectrumCollectionHandlerInterface &handler) override;

  virtual void readSpectrumCollection2(const MsRunReadConfig &config,
                                       SpectrumCollectionHandlerInterface &handler) override;

  virtual pappso::XicCoordSPtr
  newXicCoordSPtrFromSpectrumIndex(std::size_t spectrum_index,
                                   pappso::PrecisionPtr precision) const override;

  virtual pappso::XicCoordSPtr
  newXicCoordSPtrFromQualifiedMassSpectrum(const pappso::QualifiedMassSpectrum &mass_spectrum,
                                           pappso::PrecisionPtr precision) const override;

  virtual void readSpectrumCollectionByMsLevel(SpectrumCollectionHandlerInterface &handler,
                                               unsigned int ms_level) override;


  virtual std::size_t spectrumListSize() const override;

  virtual bool releaseDevice() override;

  virtual bool acquireDevice() override;

  virtual std::size_t
  spectrumStringIdentifier2SpectrumIndex(const QString &spectrum_identifier) override;


  /** @brief tells if spectra can be accessed using scan numbers
   * by default, it returns false. Only overrided functions can check if scan
   * numbers are available in the current file
   * if there are some Thermo native ids, this should be set to true
   */
  virtual bool hasScanNumbers() const override;


  /** @brief if possible, converts a scan number into a spectrum index
   * This is a convenient function to help transition from the old scan number
   * (not implemented by all vendors) to more secure spectrum index (not vendor
   * dependant).
   * It is better to not rely on this function.
   */
  virtual std::size_t scanNumber2SpectrumIndex(std::size_t scan_number) override;


  const std::vector<qint64> &getSpectrumIndexPositionInFile() const;

  virtual const OboPsiModTerm getOboPsiModTermInstrumentModelName() const override;
  virtual Trace getTicChromatogram() override;
  virtual std::vector<double> getRetentionTimeLine() override;


  /** @brief get spectrum mzML element
   * @param spectrum_index spectrum to retrieve
   * @param want_binary_data true to get peak list
   * @return shared pointer on spectrum structure
   */
  std::shared_ptr<pappso::cbor::mzcbor::Spectrum> getMzcborSpectrumSp(std::size_t spectrum_index,
                                                                      bool want_binary_data) const;

  protected:
  virtual void initialize() override;
  virtual bool accept(const QString &file_name) const override;

  virtual void fillMzcborSpectrum(std::size_t spectrum_index,
                                  pappso::cbor::mzcbor::Spectrum &spectrum,
                                  bool want_binary_data) const;

  private:
  void readSpectrumCollectionWithMsrunReadConfig(const MsRunReadConfig &config,
                                                 SpectrumCollectionHandlerInterface &handler);

  protected:
  QFileInfo m_cborFileInfo;
  std::vector<qint64> m_spectrumIndexPositionInFile;
  std::vector<qint64> m_spectrumTotalIonCountList;
  std::vector<std::uint8_t> m_spectrumMsLevelList;
  std::vector<double> m_spectrumRtList;
  std::map<QString, std::size_t> m_nativeId2SpectrumIndexMap;
  std::map<std::size_t, std::size_t> m_scan2SpectrumIndexMap;

  QFile *mpa_mzcborFileDevice = nullptr;
};
} // namespace pappso
