﻿// Copyright (c) All contributors. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

/* THIS (.cs) FILE IS GENERATED. DO NOT CHANGE IT.
 * CHANGE THE .tt FILE INSTEAD. */

/*
    **** REFERENCE BLOG POST ****
    About SIMD(Single Instruction Multiple Data). See: https://en.wikipedia.org/wiki/SIMD
    In August 2020, .NET Core 3.1 provides the `Hardware Intrinsics` for x86/x64 processor.
        Blog Post : https://devblogs.microsoft.com/dotnet/hardware-intrinsics-in-net-core/
        API Document : https://docs.microsoft.com/ja-jp/dotnet/api/system.runtime.intrinsics.X86
    Incoming .NET 5 will provide the Hardware Intrinsics API for Arm processor.
        Blog Post : https://devblogs.microsoft.com/dotnet/announcing-net-5-preview-4-and-our-journey-to-one-net/#user-content-arm64-performance
        API Document : https://docs.microsoft.com/ja-jp/dotnet/api/system.runtime.intrinsics.Arm
*/

/*
    In x86/x64, there are several streaming SIMD extensions.
        Official Site: https://software.intel.com/sites/landingpage/IntrinsicsGuide/

    .NET Core 3.1 only provides up to AVX2 streaming SIMD extension.
    AVX2 is supproted since Intel's Haswell micro architecture(2013/6).
    Sse4.2 and PopCnt are supported since Intel's Nehalem micro architecture(2008/11).

    Because of a problem in serializing MessagePack, I implemented all of them using SSE4.2 features instead of AVX2.
    In MessagePack, the size of the resultant encoded numeric value varies according to the range to which the numeric value belongs.

    |            max excluded            |              min excluded              | encoded byte length | MessagePack code |
    | ---------------------------------- | -------------------------------------- | ------------------- | ---------------- |
    |           short.MinValue           |            int.MinValue - 1            |          5          |      Int32       |
    | MessagePackRange.MinFixNegativeInt |           sbyte.MinValue - 1           |          2          |       Int8       |
    |           sbyte.MinValue           |           short.MinValue - 1           |          3          |      Int16       |
    |         sbyte.MaxValue + 1         | MessagePackRange.MinFixNegativeInt - 1 |          1          |      FixNum      |
    |          byte.MaxValue + 1         |             sbyte.MaxValue             |          2          |      UInt8       |
    |         ushort.MaxValue + 1        |             byte.MaxValue              |          3          |      UInt16      |
    |          uint.MaxValue + 1         |            ushort.MaxValue             |          5          |      UInt32      |

    We will insert appropriate MessagePackCode with appropriate reordering of the input byte sequence to correspond to the variable-length output results.
    In SIMD programming circles, "reordering" is referred to as "shuffle" or "permute".

    MessagePack serialization using SIMD consists of the following steps.

    - Load the Input Values.
    - Classify the Input Values.
    - Calculate Total Output Byte Count.
    - Get the destination span from writer with the Total Output Byte Count.

    In subsequent steps, the Input Values are divided into halves and treated as the Input Values.
    There are 16 input values in the sbyte type, 8 in the short type, and 4 in the int type.

    - Shuffle the Input Values to make the Output Byte Sequence.
    - Place the appropriate MessagePack Codes in the Output Byte Sequence according to the classification.
    - Store the Output Byte Sequence in the destination span.
*/

/*
    How to classify the input values?

    Conditional branching is not favored in SIMD programming.
    The technique of Shader programming is similar to that of SIMD.

    I use only Sse2.CompareGreaterThan(left, right). See : https://software.intel.com/sites/landingpage/IntrinsicsGuide/#text=_mm_cmpgt_epi8&expand=915

    public static System.Runtime.Intrinsics.Vector128<sbyte> CompareGreaterThan (System.Runtime.Intrinsics.Vector128<sbyte> left, System.Runtime.Intrinsics.Vector128<sbyte> right);

    Pseudo Code:
        System.Runtime.Intrinsics.Vector128<sbyte> CompareGreaterThan (System.Runtime.Intrinsics.Vector128<sbyte> left, System.Runtime.Intrinsics.Vector128<sbyte> right)
        {
            Vector128<sbyte> answer = default;
            sbyte* leftPointer = (sbyte*)&left;
            sbyte* rightPointer = (sbyte*)&right;
            sbyte* answerPointer = (sbyte*)&answer;
            for (int i = 0; i < 16; i++)
            {
                if (*leftPointer++ > *rightPointer++)
                {
                    *answer++ = -1;
                }
                else
                {
                    *answer++ = 0;
                }
            }

            return answer;
        }

    If the condition is true, the corresponding bit intervals are all 1.
    This state where all the bits are 1 is sometimes regarded as -1, and sometimes it is simply used as a bit mask.
*/

/*
    What is the `shuffle`?
    See : https://software.intel.com/sites/landingpage/IntrinsicsGuide/#text=_mm_shuffle_epi8&expand=5153
    See : https://www.officedaytime.com/simd512e/simdimg/si.php?f=pshufb

    public static System.Runtime.Intrinsics.Vector128<byte> Shuffle (System.Runtime.Intrinsics.Vector128<byte> value, System.Runtime.Intrinsics.Vector128<byte> mask);
    `shuffle` packed 8-bit integers in value according to shuffle control mask in the corresponding 8-bit element of mask, and returns.

    Example(pseudo code):
        Vector128<byte> input   = Vector128.Create(0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10);
        Vector128<byte> shuffle = Vector128.Create(0x80, 0, 15, 0x80, 2, 2, 3, 0x80, 0x80, 14, 13, 12, 0, 5, 1, 6);
        Vector128<byte> result = Ssse3.Shuffle(input, shuffle);

        Console.WriteLine(result.ToString());
        // <0, 0x01, 0x10, 0, 0x03, 0x03, 0x04, 0, 0, 0x0F, 0x0E, 0x0D, 0x01, 0x06, 0x02, 0x07>
*/

using System;

#pragma warning disable SA1649 // File name should match first type name
#pragma warning disable CS0436 // The same name of the primary package

namespace MessagePack.Formatters
{
    internal static class SingleInstructionMultipleDataPrimitiveArrayFormatterHelper
    {
        /*
            Sse2.MaskMove(src, mask, dest);
            Vector128<byte> is a byte array consists of 16 elements.
            This function stores the byte element of src at the corresponding position of mask if it is greater than or equal to 0x80 and does nothing if it is less than that.

            See : https://software.intel.com/sites/landingpage/IntrinsicsGuide/#text=maskmove&expand=3555
        */

        /// <summary>
        /// Gets Mask Vector Table.
        /// MessagePack SIMD Serialization needs to store the output bytes to the destination memory.
        /// <example>
        /// <code>
        /// /* Vector128&lt;byte&gt; vector128; int outputLength; byte* destination; */
        /// var maskTable = SingleInstructionMultipleDataPrimitiveArrayFormatterHelper.StoreMaskTable;
        /// fixed (byte* maskTablePointer = &amp;maskTable[0])
        /// {
        ///     var mask = Sse2.LoadVector128(maskTablePointer + (outputLength &lt;&lt; 4));
        ///     Sse2.MaskMove(vector128, mask, destination);
        /// }
        /// </code>
        /// </example>
        /// </summary>
        public static ReadOnlySpan<byte> StoreMaskTable => new byte[17 * 16] { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, };
    }

    public sealed partial class SByteArrayFormatter
    {
        /*
            1(2) : -32 > i >= short.MinValue
            0(1) : sbyte.MaxValue >= i >= -32

            2 ^ 8 == 256

            Values of sbyte type can be classified into two types by MessagePack Code(FixNum and Int8).
            If 16 Input Values are all in the range of FixNum, they can be just copied to the destination span.
            If not, `shuffle` is needed.

            Vector<byte> ShuffleAndMaskTable[256].
            C# doesn't allow the above expression.
            So ShuffleAndMaskTable is written as byte[4096].

            Sse2.CompareGreaterThan(-32, i) can classify them.

            Sse2.MoveMask gathers highest bits of Vector128<sbyte>.
            In other words, this API compiles the comparison results of 16 sbyte numbers into 16 bits.
            Lower 8bits represents the results of the first 8 sbyte values.

            Total Output Byte Sequence Length can be calculated by Popcnt.PopCount static method.
                See : https://docs.microsoft.com/en-us/cpp/intrinsics/popcnt16-popcnt-popcnt64?view=vs-2019

            Since there is only one type of MessagePack code(==Int8) in the case of sbyte, I used Sse41.BlendVariable static method to synthesize the Output Byte Sequence.

            Pseudo Code:
                public static System.Runtime.Intrinsics.Vector128<sbyte> BlendVariable (System.Runtime.Intrinsics.Vector128<sbyte> left, System.Runtime.Intrinsics.Vector128<sbyte> right, System.Runtime.Intrinsics.Vector128<sbyte> mask)
                {
                    Vector128<sbyte> answer = default;
                    sbyte* leftPointer = (sbyte*)&left;
                    sbyte* rightPointer = (sbyte*)&right;
                    sbyte* maskPointer = (sbyte*)&mask;
                    sbyte* answerPointer = (sbyte*)&answer;
                    for (int i = 0; i < 16; i++)
                    {
                        bool shouldSelectLeft = mask[i] >= 0;
                        if (shouldSelectLeft)
                        {
                            answer[i] = left[i];
                        }
                        else
                        {
                            answer[i] = right[i];
                        }
                    }

                    return answer;
                }
        */

        private static ReadOnlySpan<byte> ShuffleAndMaskTable => new byte[256 * 16] { 0, 1, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 7, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 7, 0x80, 0x80, 0, 1, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 7, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 7, 0x80, 0, 1, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 6, 0x80, 7, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 6, 0x80, 7, 0x80, 0, 1, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 5, 0x80, 6, 0x80, 7, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0x80, 0, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 0x80, 1, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0x80, 0, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, 0x80, 0x80, 0, 0x80, 1, 0x80, 2, 0x80, 3, 0x80, 4, 0x80, 5, 0x80, 6, 0x80, 7, };
    }

    public sealed partial class Int16ArrayFormatter
    {
        /*
            0(3) : sbyte.MinValue > i >= short.MinValue
           -1(2) : MessagePackRange.MinFixNegativeInt > i > sbyte.MinValue - 1
           -2(1) : sbyte.MaxValue + 1 > i > MessagePackRange.MinFixNegativeInt - 1
           -3(2) : byte.MaxValue + 1 > i > sbyte.MaxValue
           -4(3) : ushort.MaxValue >= i > byte.MaxValue

            5 ^ 4 == 625
        */
        private const int CountTableOffset = 625 * 32;

        private static ReadOnlySpan<byte> ShuffleAndMaskTable => new byte[625 * 36] { 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 3, 2, 0x80, 5, 4, 0x80, 7, 6, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 4, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 8, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 9, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0, 11, 0, 0, 0, 10, 0, 0, 0, 11, 0, 0, 0, 12, 0, 0, 0 };
    }

    public sealed partial class Int32ArrayFormatter
    {
        /*
             0(5) : short.MinValue > i >= int.MinValue
            -1(3) : sbyte.MinValue > i >= short.MinValue
            -2(2) : -32 > i >= sbyte.MinValue
            -3(1) : sbyte.MaxValue + 1 > i >= -32
            -4(2) : byte.MaxValue + 1 > i > sbyte.MaxValue
            -5(3) : ushort.MaxValue + 1 > i > byte.MaxValue
            -6(5) : int.MaxValue >= i > ushort.MaxValue

            7 ^ 2 == 49
        */
        private const int CountTableOffset = 49 * 32;

        private static ReadOnlySpan<byte> ShuffleAndMaskTable => new byte[49 * 36] { 0x80, 3, 2, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, MessagePackCode.UInt8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, MessagePackCode.UInt16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int32, 0, 0, 0, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int16, 0, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.Int8, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt8, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt16, 0, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x80, 3, 2, 1, 0, 0x80, 7, 6, 5, 4, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, MessagePackCode.UInt32, 0, 0, 0, 0, MessagePackCode.UInt32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 10, 0, 0, 0, 8, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 4, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 5, 0, 0, 0, 4, 0, 0, 0, 3, 0, 0, 0, 4, 0, 0, 0, 5, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 4, 0, 0, 0, 3, 0, 0, 0, 2, 0, 0, 0, 3, 0, 0, 0, 4, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 5, 0, 0, 0, 4, 0, 0, 0, 3, 0, 0, 0, 4, 0, 0, 0, 5, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 6, 0, 0, 0, 5, 0, 0, 0, 4, 0, 0, 0, 5, 0, 0, 0, 6, 0, 0, 0, 8, 0, 0, 0, 10, 0, 0, 0, 8, 0, 0, 0, 7, 0, 0, 0, 6, 0, 0, 0, 7, 0, 0, 0, 8, 0, 0, 0, 10, 0, 0, 0 };
    }

    public sealed partial class SByteArrayFormatter
    {
        public static readonly SByteArrayFormatter Instance = new SByteArrayFormatter();

        private SByteArrayFormatter()
        {
        }

        public SByte[]? Deserialize(ref MessagePackReader reader, MessagePackSerializerOptions options)
        {
            if (reader.TryReadNil())
            {
                return default;
            }

            var len = reader.ReadArrayHeader();
            if (len == 0)
            {
                return Array.Empty<SByte>();
            }

            var array = new SByte[len];
            for (var i = 0; i < array.Length; i++)
            {
                array[i] = reader.ReadSByte();
            }

            return array;
        }
    }

    public sealed partial class Int16ArrayFormatter
    {
        public static readonly Int16ArrayFormatter Instance = new Int16ArrayFormatter();

        private Int16ArrayFormatter()
        {
        }

        public Int16[]? Deserialize(ref MessagePackReader reader, MessagePackSerializerOptions options)
        {
            if (reader.TryReadNil())
            {
                return default;
            }

            var len = reader.ReadArrayHeader();
            if (len == 0)
            {
                return Array.Empty<Int16>();
            }

            var array = new Int16[len];
            for (var i = 0; i < array.Length; i++)
            {
                array[i] = reader.ReadInt16();
            }

            return array;
        }
    }

    public sealed partial class Int32ArrayFormatter
    {
        public static readonly Int32ArrayFormatter Instance = new Int32ArrayFormatter();

        private Int32ArrayFormatter()
        {
        }

        public Int32[]? Deserialize(ref MessagePackReader reader, MessagePackSerializerOptions options)
        {
            if (reader.TryReadNil())
            {
                return default;
            }

            var len = reader.ReadArrayHeader();
            if (len == 0)
            {
                return Array.Empty<Int32>();
            }

            var array = new Int32[len];
            for (var i = 0; i < array.Length; i++)
            {
                array[i] = reader.ReadInt32();
            }

            return array;
        }
    }

    public sealed partial class SingleArrayFormatter
    {
        public static readonly SingleArrayFormatter Instance = new SingleArrayFormatter();

        private SingleArrayFormatter()
        {
        }

        public Single[]? Deserialize(ref MessagePackReader reader, MessagePackSerializerOptions options)
        {
            if (reader.TryReadNil())
            {
                return default;
            }

            var len = reader.ReadArrayHeader();
            if (len == 0)
            {
                return Array.Empty<Single>();
            }

            var array = new Single[len];
            for (var i = 0; i < array.Length; i++)
            {
                array[i] = reader.ReadSingle();
            }

            return array;
        }
    }

    public sealed partial class DoubleArrayFormatter
    {
        public static readonly DoubleArrayFormatter Instance = new DoubleArrayFormatter();

        private DoubleArrayFormatter()
        {
        }

        public Double[]? Deserialize(ref MessagePackReader reader, MessagePackSerializerOptions options)
        {
            if (reader.TryReadNil())
            {
                return default;
            }

            var len = reader.ReadArrayHeader();
            if (len == 0)
            {
                return Array.Empty<Double>();
            }

            var array = new Double[len];
            for (var i = 0; i < array.Length; i++)
            {
                array[i] = reader.ReadDouble();
            }

            return array;
        }
    }

    public sealed partial class BooleanArrayFormatter
    {
        public static readonly BooleanArrayFormatter Instance = new BooleanArrayFormatter();

        private BooleanArrayFormatter()
        {
        }
    }
}
