; Tests of the SHA-384 formalization
;
; Copyright (C) 2013-2019 Kestrel Institute
;
; License: A 3-clause BSD license. See the file books/3BSD-mod.txt.
;
; Author: Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "SHA2")

(include-book "sha-384")
(include-book "kestrel/bv-lists/string-to-bits" :dir :system)
(include-book "kestrel/utilities/hex-string-to-bytes" :dir :system)
(include-book "std/testing/assert-equal" :dir :system)

;;
;; Examples (see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA384.pdf)
;;

;; Example 1, bit version
(acl2::assert-equal (sha-384 (acl2::string-to-bits "abc"))
                    (acl2::hex-string-to-bytes! "CB00753F45A35E8BB5A03D699AC65007272C32AB0EDED1631A8B605A43FF5BED8086072BA1E7CC2358BAECA134C825A7"))

;; Example 2, bit version
(acl2::assert-equal (sha-384 (acl2::string-to-bits "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"))
                    (acl2::hex-string-to-bytes! "09330C33F71147E83D192FC782CD1B4753111B173B3B05D22FA08086E3B0F712FCC7C71A557E2DB966C3E9FA91746039"))

;; Example 1, byte version
(acl2::assert-equal (sha-384-bytes (acl2::string-to-bytes "abc"))
                    (acl2::hex-string-to-bytes! "CB00753F45A35E8BB5A03D699AC65007272C32AB0EDED1631A8B605A43FF5BED8086072BA1E7CC2358BAECA134C825A7"))

;; Example 2, byte version
(acl2::assert-equal (sha-384-bytes (acl2::string-to-bytes "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"))
                    (acl2::hex-string-to-bytes! "09330C33F71147E83D192FC782CD1B4753111B173B3B05D22FA08086E3B0F712FCC7C71A557E2DB966C3E9FA91746039"))

;;
;; Tests from the .zip file https://github.com/coruus/nist-testvectors/raw/master/csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip:
;;

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! ""))
                    (acl2::hex-string-to-bytes! "38b060a751ac96384cd9327eb1b1e36a21fdb71114be07434c0cc7bf63f6e1da274edebfe76f65fbd51ad2f14898b95b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "c5"))
                    (acl2::hex-string-to-bytes! "b52b72da75d0666379e20f9b4a79c33a329a01f06a2fb7865c9062a28c1de860ba432edfd86b4cb1cb8a75b46076e3b1"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "6ece"))
                    (acl2::hex-string-to-bytes! "53d4773da50d8be4145d8f3a7098ff3691a554a29ae6f652cc7121eb8bc96fd2210e06ae2fa2a36c4b3b3497341e70f0"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1fa4d5"))
                    (acl2::hex-string-to-bytes! "e4ca4663dff189541cd026dcc056626419028774666f5b379b99f4887c7237bdbd3bea46d5388be0efc2d4b7989ab2c4"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "50e3853d"))
                    (acl2::hex-string-to-bytes! "936a3c3991716ba4c413bc03de20f5ce1c63703b3a5bdb6ab558c9ff70d537e46eb4a15d9f2c85e68d8678de5682695e"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "4b5fab61e0"))
                    (acl2::hex-string-to-bytes! "fb390aa5b70b068a54d6d5127df6a6227becc4d6f891fd3f6068b917a883c9b66f318fddb6384d10be8c7af0d3132f03"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "dad95a4b4d37"))
                    (acl2::hex-string-to-bytes! "3a2b40f453925bc3ce17d640757ee0e899390b4a8d984d0297c1bae6b60b9f2603bf71c323fd171011372335e5702e40"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "121835fe3700b7"))
                    (acl2::hex-string-to-bytes! "7bd06a94acba7beb3c5a9b9e8769c3da6691c482d78b1e5c7619b36630eba4e596d11c410a4c87006f4716b6f17bb9a0"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "de60275bdafce4b1"))
                    (acl2::hex-string-to-bytes! "a3d861d866c1362423eb21c6bec8e44b74ce993c55baa2b6640567560ebecdaeda07183dbbbd95e0f522caee5ddbdaf0"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "8d45a55d5ce1f928e6"))
                    (acl2::hex-string-to-bytes! "de76683575a050e2eb5ef95ee201f82416478a1d14bf3d96d1fd4efd52b1a28fed8dfee1830070001dc102a21f761d20"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "5c7dde9b3894d73cefe1"))
                    (acl2::hex-string-to-bytes! "f31b22115fa7178e78223e06aae870547ab93c6eb3c3910b0ee16e6106db55935d6c0eb820132a2078ece1067efc81c3"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "967fa34c07e4945a77051a"))
                    (acl2::hex-string-to-bytes! "f8f24d81c4f8f23ecb42d76ed5d2b34c9cbc1f0a97234d1114804b5999759f3131c741d5768cc928163503c5f55f594b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "2209112ee779bf6d95711105"))
                    (acl2::hex-string-to-bytes! "09c54bf533a26c7447caa5783db2ec7ef5e55752da7f2a2c4e360982a94ec1ca2cb6a157d34eed28de978b4145e17ebc"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "20442e1c3f3c88919c39978b78"))
                    (acl2::hex-string-to-bytes! "50bc95b036e0f54d833032a80d45c2ac38b3d29e9c7f72a2eb14781e9241d2a4b8e8dba6ee6f4c9e46a758d5712dbd39"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "4d43702be4f0530319555d7f1a33"))
                    (acl2::hex-string-to-bytes! "83c8f0bb762801eb26cc5115abebb657c18ff811de500b32b7a568a220a287e900b6c75224fe7429169fbd534cb588e1"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "5e2a79a544af85f150f7a9d209fd44"))
                    (acl2::hex-string-to-bytes! "8051ebc9cabb052cabe07e4023e2140808b77d25b07b96d2e3c22393f71b116c1a1e41bf62e57f73ff67871da7c93cf9"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e1bb967b5d379a4aa39050274d09bd93"))
                    (acl2::hex-string-to-bytes! "3b04f96965ad2fbabd4df25d5d8c95589d069c312ee48539090b2d7b495d2446c31eb2b8f8ffb3012bdce065323d9f48"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "bb84a014cd17cc232c98ae8b0709917e9d"))
                    (acl2::hex-string-to-bytes! "85227ae057f2082adf178cae996449100b6a3119e4c415a99e25be6ef20ba8c0eae818d60f71c5c83ff2d4c59aa75263"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "c3411a0592f1f4fa698815238997db356418"))
                    (acl2::hex-string-to-bytes! "233ac44170d9f452a1a0231622030b15c104ff8ecaa3fccdb9e9e5031fd5b4220186a8edd032849c8b93dc183a5c8627"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e831b739e8eb9f787f63c0bb071ddcc9f44cab"))
                    (acl2::hex-string-to-bytes! "91722d4b7aecc211bb8a5480c6855f3b71be4647e1dde0380c23afaa03f45c642606a24506e0317bf51506a483de28ac"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b8a7bbccde46e85f1223237d9353b78c3b19727b"))
                    (acl2::hex-string-to-bytes! "28ba69dc05e6e29de91924114d6c9fc7612f6d2a68b07fa001df059bcf98f7aa85389caeb966eaa299c79fe1fd1e40e3"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "cf391b8aabec6f81288c8b7b92843be23d2e847574"))
                    (acl2::hex-string-to-bytes! "121e5ef697df491a53d7bae121416aa653d759a37db9d0b993031b18a0ef160ed98842a291e1ba2cea8b998bc5eee0b1"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "9d65d88bffed764c286f34894f991600d1a67d622382"))
                    (acl2::hex-string-to-bytes! "84b6e0d6a45329daf47a793418ed5dbde01336b4b9468bb69e5da61c42b691e6794e6ed0e8fb1b8e7d4cd3cbaadc520a"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "bab6ea46fb717f73f0628132a2b96be383774f1e5d8b6d"))
                    (acl2::hex-string-to-bytes! "e969aca1b50e928cad29a0d756457f6de8d7a4e589fd41e53a1e758c3b20f9b81b36bf098a49102fbf869651ca9a98b5"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "8853b00e869764adb527a07b073c85a24e6c208ba47eef4e"))
                    (acl2::hex-string-to-bytes! "09ad44e85ac190e2d1c3ceb4efbea10ded34d0de961fe4ee268132c48e38660e6cf585bfffb8f7b00b0fad1514312b63"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "71fe1ba5d299495d2a56039c64032ec6263d437f55e3f5bedb"))
                    (acl2::hex-string-to-bytes! "b41a5d3b4af6d4b9c349e0788538e9a0311086894df7b72cf5aaf4091a7e039e4e89cc77a123474e6d1bac438e5e9f88"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "4ebe07d03c93e849b4bbfe9f2d2294bf6ccab457f76d3f99cb64"))
                    (acl2::hex-string-to-bytes! "beba464d7065185587fad89bfcea9635bf0ab775c3eb8c147b5b2bd8601db6dba0590b50dd1068733f20dc68e004a090"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "918ac0a97ec1632908489e5242ba284bc811aa7197242cf7226fcd"))
                    (acl2::hex-string-to-bytes! "c4baf6397a4c6e26492b63a4aab7dffdd0051d8f51938ac24cfd8dae2f7afed1a4aa2430d7aeb0be2a72b21a6c50198c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "97477f7272d8a89e38e796c533e9f8a8ae4c92ccaa5d907ed26a1a15"))
                    (acl2::hex-string-to-bytes! "d1ad524ebe908d7c5aff50e6cb780fd3a70e87c914a36b93c4e35f5b2cb03850b122b480ef8587d4a44f22467f4c480c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "718e0cfe1386cb1421b4799b15788b862bf03a8072bb30d02303888032"))
                    (acl2::hex-string-to-bytes! "6d8b8a5bc7ea365ea07f11d3b12e95872a9633684752495cc431636caf1b273a35321044af31c974d8575d38711f56c6"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "d3b07f0fd5d4cd3188aead8dc8338de42056e2e8487eca51ec37ef2daf27"))
                    (acl2::hex-string-to-bytes! "adcc2e954c91db3db2d71d0dee1f030e723bee1a23816fe003ac5dc862a0872ef51ff386c18be6ebcaa493f32d1195b9"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "695b9efe1809abd5d44eae957ddf9c2cd3c75fae2f522855712a07c639c0b9"))
                    (acl2::hex-string-to-bytes! "3bb95d164d94595a1187f77fc26c280ffbb08e74ec7947aa3e5b38bec7c6f8115c4d880788c2402dbb3e5b94afd130ee"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "be01e520e69f04174ccf95455b1c81445298264d9adc4958574a52843d95b8ba"))
                    (acl2::hex-string-to-bytes! "c5cf54b8e3105b1c7bf7a43754d915b0947f28b6dc94a019182929b5c848e11441c9e4e90c7449f4c3cd12954f0f5d99"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "98ef7301f957d73d4e821d5873e8a9b5970fbd219d5cf74ec2291b8381181391b4"))
                    (acl2::hex-string-to-bytes! "b2564bbb159c3aeadbae0e4a4437f7c5111020e9ad0f4eb508147a961ac22a01e1a26df046b14ee5e8a49d9ed22b8cd1"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "873ba7f8b71517ec50297b21cf94cdb7a58abbb8829206f0d3f328ff8a6939ac1d1d"))
                    (acl2::hex-string-to-bytes! "7405fdd557d2b5d4f65e9e85f508d3791495f1820d37cabfc8dbb74d7b41df8613d995c612d378dd88337e00951d0280"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e3bd4bc3a60cddd26c20aa86364bd44f4a07f3302825ad0ac127881de4eafbccf988cb"))
                    (acl2::hex-string-to-bytes! "199d5423a0e26438f4cea0081a89e0b6c84ca93f7c3120c8104b51c6edc04e0f6a203bb77d59973a7411a0efbe93a09d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "8716e4b86deff9da2a8ed55baa43582a7586ec9cd38ac3a933156158cd8e5b7887585e91"))
                    (acl2::hex-string-to-bytes! "0d2306d9c0a8ce57bc7869b439376c07ce352a41d86ab6cf4a5654cccd5c724fe1b62b2c1101c986222f5264ab3fdd66"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "f8a503aaa25ef2cea25e31930c3a90db468cd3a862f4a93aab5de2777e82dc905cb03ee23c"))
                    (acl2::hex-string-to-bytes! "773ee958fe93dfd1b73af24d27ddce33144a9249d5a671682a56df30d0bbf92b9327130022075185d396de752959304f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "669025175ea917cdd7a71ff4ec0c45bf1246d2a6f031c00b71de701e17939bfe92128b21911e"))
                    (acl2::hex-string-to-bytes! "9ff6be3f02c7c5d0206f4b944c0843cb68bea8f9b7c8cc0b729503db5005c7cd5cb14e3457d8f5eabf733fca9084f16b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b35fb2262edfa14938a0fba03eb2a25d377974b11f556491a781d0ba2b3c0ff3e42749925fef8b"))
                    (acl2::hex-string-to-bytes! "835b05a4bf00c2594c3c8c13da6c273a0d9efdea0da72b71b19d326bf5ce968c2e577a7d99fc0f985afd23b46423129d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "9d86b45df8d7dae0cf6b0bc208666ee1163a39e6116d6d240c9dc1c3a3c1db1dd3b1c6680fe9a196"))
                    (acl2::hex-string-to-bytes! "a84c469c24696f81d7df4ee8cd76aa584f8c9960eaa9908d3e3ebc5eea7d0b50afdded39deb94fd59323a21a6539e93f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "414f5619f6dfd45853bbabd224cc305d77350ad253358910a74f3a4381a9b86680b3c4068c089849c4"))
                    (acl2::hex-string-to-bytes! "848d481e3bbf5dd726f625cf6a444d995b36262c9f80d583b77accf1707e3f49bb3dc480a560694d769aa1ce65d69428"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e2658324821ae7b0faa0cdd63ee9efb9fcbe82092d04696feb3da92c82521dfdc98f6b41b3ef365d219a"))
                    (acl2::hex-string-to-bytes! "3ea5d0799f1a4dcab9149a40ab74bec9c8d76d8e392c1e63e080ddec2ec535f80be9f00927be281ec97ac0c882bb0bbf"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "7e80271bb5f2cc7ddae4158658e4e8d06e04a39385da0ecac1cb8e91d68a9bd21ddb7320e79d10e3110758"))
                    (acl2::hex-string-to-bytes! "fa00bc0359a642dcb3559656094eb2fd4f63bc57f0d34abff26df5c54cc63dbeb4eac75905296e7fb69f871e134083f6"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "43d2828e86f7856b78c66cfa3d602387c290975afd021a8b76af0918069cac35dec45de3cc52edc4ba14432e"))
                    (acl2::hex-string-to-bytes! "6c9e367e066032ce47ba2575565932002cc786f533c5551656abfe7391e7dcb5f9d9e047adace23d32f8acedfd0cafc5"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "3f49bb645cced7530b8b82e6cf07fbf670f7ef0ba0583d16debafc639bdfbfc99b8417249f7f5a05410aa3a71f"))
                    (acl2::hex-string-to-bytes! "2b301a14647a67429cc3e7da02c4093a739640f7b47a305251d2855e75e09e60e262b279a073077d1fb6d0f04788f2b8"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "31aac06a59b74bf478617c1637fa6c5593df168b8d58b1e923bf3e3d80e55d7170b16454160ab29ee1f7412ebc05"))
                    (acl2::hex-string-to-bytes! "ddd245c9b29ceac60506fb6bd6e8037889cb73d6ecc669fd129060a8a8f58971ac572d3ec9b44404f81381d0fd35a649"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "c10b2852054d8034e07906c7fce3ce99402321a648bb881f13fb276afc224c6aecc64800cd767ed2429db94b95a9c3"))
                    (acl2::hex-string-to-bytes! "a44640fb4ce6dfd4a10290a0aecdb453054a9b54f2583e97bb7dc2b005e5fa2b4fda17b1f75902f51c18c0caad35833c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b1eeef324b499f19eba322215fe3ce19c9f000b698d2b2dab7145015046cc86d049ee15ad59dcd1564f30112e06444cb"))
                    (acl2::hex-string-to-bytes! "38742d18bfa6e918b888d68d1034e61f65dec0759172c2dbf08cf1e132b217eaf4ec29e15db7f4b07e08a70cc5662012"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "790dbba09965c9774dd60a32e010c50d6d518968a220141dc33e7410f2da6c08ad0419bd9864d5327d2c5c44914b2e83f9"))
                    (acl2::hex-string-to-bytes! "9174958bc8f4ed4731eced999bea2f63032f52bc8c46bcd903232f3fbc5046f0d6c203d43a078b822fc05101404f2635"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "f7b577f1396b23c27eb637e53d3d92460270b001cc612fd3b4d68bcdd09c2d50571ea4350636324cc2428a087e7bd8785f82"))
                    (acl2::hex-string-to-bytes! "80afe111e44ad9aff9e39c4cf9e6b4c520072b4550e62b1740160a04f8d530612dc098917a556b44977d0e73df518bee"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "7370d9b453936955b9c9d336f4b283237986232de007bf412fb426ff5b4093c80c428c19a12e0b187484dc6d5f4746537fb1ed"))
                    (acl2::hex-string-to-bytes! "6cd29159820d34e5706dd867e6363758d312660d4daca8d2abf677f234746e97a0a6224eb054066a0b74e18c70965368"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e8620170f0f39328bdf8888148cfd17730f314ea68d8fea02d16d98a3cca61484139d3ee92b748091dc841dda08483f1184025ce"))
                    (acl2::hex-string-to-bytes! "29c408a6a5045f397b56dfb5292c7c16028c63f068e699b86a891ff8501208ec9398dbaf463c00f39af7b2cbe45bac15"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "75d4216bad77943bfe82be216157843b0da0fd16eeee847153a00b43e707b2ffe2c898168081f0bdb3af58f214cf67b920c385aca1"))
                    (acl2::hex-string-to-bytes! "0170357385a20865a8d30c2df39406903ff88c7f70fa1a7a5aaa86711d64046c432a1b139668ae5abed637d1dc4107b7"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "2fa90c2210e3096caed122b74eb9559977120e5d9a97eeb3f99bcba6c19cf8cf791ac6c8a0a94ae49246611dace7f24eb05673a36b3e"))
                    (acl2::hex-string-to-bytes! "6c2ced3fae94dbd92f4170b63f1ff6fcd8194f60937b22b4f3c95fc9e104b77148f9bc6c16a890de86d9ef1554c96fa0"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "a8de55170c6dc0d80de32f508bf49b7046aeea4b975a511b5ea3dc853bfa4b1e0138202d6785f6b3017914a86f824288d586dc0e8bc924"))
                    (acl2::hex-string-to-bytes! "2bc3b10c148200f7919b57afe1d7db773ffd235e04fec6897dd94f13ad9c437ef50900a40937f82a39daf2aa2b3dfd68"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "accd9d05fb7ef3043470836137554af117440b3ccca7a280285494f90dfaea60dcbf40b230271932cd3875b1d3dca60d38865ff874180efa"))
                    (acl2::hex-string-to-bytes! "b9e02df93e9254180d6a15288d77088b5a5ce517644109b4e2532be315f08dee43491608a1127dcdf69397406e23d231"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "32b835c180cc4006c11a61c65b03c099510e1d4f0a94b63d54bd6bd5a8ab207ab0f4639206564edc3fa6af03280a67744f68106dc51ee35723"))
                    (acl2::hex-string-to-bytes! "df97a1c5dda6f9dde749f213e429db84f0dcd481d43bf58e6142968d629ecf05b262830a7dac87f67f4383975f3e821d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "9409f9efadbf190b253367629f8f368c9d5ac262e94ab86f3559f9a1fe1a9b44b64e313121b34d43001c430bedc62fc586ea398acd8f17c7cfa2"))
                    (acl2::hex-string-to-bytes! "e1a69388ee6b6d234108ec29402cd0afd74957d990c7bdb544cf11e8eb2ccd170b6b5a74431be70364d7a31b926ff53c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "289586baf8daced50db14c936529a0a6438eb5da8d3d469172b6a06f4ff3a956d4f9219563ac285cb8e70074cfcc152c02593a97733c36f4a9e97f"))
                    (acl2::hex-string-to-bytes! "509e996c1e11611c243021b8b78f2ad90c5a9263bbf35910db7c8ec102aa7c518066fff8ce88562fec2cd6dfe04056ae"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "156530cd6ed3baf1fd7232c7ff204f3c7d4d106016afa3bdff3786e84843ec556115626fdc84b2e874f1074e4f7d53e08079ee9fd01f80a8be7f20c0"))
                    (acl2::hex-string-to-bytes! "7b8a598029ca0ed475a72c0644ac81c63d72fd51305dada07b0ab4a29e47422f59e12643179269ca3d7d10446b372b2c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "30655a6b5a5965db992e7248d24141055e988d726abb8e729dc5c21ffcbaedbc0b1b5fea35b8751f6ec6625517312fff2234014176269b60959723787c"))
                    (acl2::hex-string-to-bytes! "cfaf443e95deeb3cc1910771a2c0692a54b18b3633dc5414e71ae08877f0804818f67f7196c52e26b762dd12bb7a86ca"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "345c3c022e20144e135604078762ef5c8a8f038cf1b1d6a91709b59dd068396a9e971ab628f74886e765384a23607c1a1e6e973f8fbb0ff55104c46f5db3"))
                    (acl2::hex-string-to-bytes! "bfb1d5ee3a0b629058ecc521c706c2f9241c48cda3dcfdba660a2601d832a7a872a2bb840f3b98d21c37e28f9041a5b2"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "0b94a0f43a92408963a59ded01a9338283a6ff1bfbacd9051a0104445c7f041e8037afde3b5a87d22d5a4796144cbc944091d6cc47b5ffd1f997ab1496ca31"))
                    (acl2::hex-string-to-bytes! "07a045c9590e9901d0598e604c4649554a823de996fa438cc81a634344eeb98e5f3c0c234ba30e2285a4d7ab568f2610"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "93035d3a13ae1b06dd033e764aca0124961da79c366c6c756bc4bcc11850a3a8d120854f34290fff7c8d6d83531dbdd1e81cc4ed4246e00bd4113ef451334daa"))
                    (acl2::hex-string-to-bytes! "8d46cc84b6c2deb206aa5c861798798751a26ee74b1daf3a557c41aebd65adc027559f7cd92b255b374c83bd55568b45"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "bfb94dfbe0d9a509b78d164a722050054dad91c9a8e260545d037eb450321aac48ed4459fdd8a41572bd6c9c84d18b6ec339247482cc3ee52a1bbd6bd4ae918216"))
                    (acl2::hex-string-to-bytes! "13af0be02986ea3176e8c65534ec9f32c23b53c93a73b15c26b9ecbd8a1181ae184a372e9f5e0596cd6606849aeae8e0"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1c8924a16fa7c602aff5ee961798bd44fe53798bf44c3d6b0d13ef837de07377651c1e94ed236eb79349f86ac31ba151a7e711c5407e65beb63824f6ec39754b58f3"))
                    (acl2::hex-string-to-bytes! "5be6482851ddafde582f2804071a702ae39bacb688741b7c37bbae99821ce4d3f47d5b097fd8eefa0ef9248a34f5d3ce"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "184215145da49db417e8bdd573d6282de073e674c3dea86b6c78591d4947f5655a9d9eb0e5f4ed046b1d8642da9aefa80448a299504160a1b000c9b4d3c62ab69b3d96"))
                    (acl2::hex-string-to-bytes! "8995cd7fc0956e124075440686beece17a6256b282e7988a0c998f790e3995c974383179893477bcc32d1f114129b496"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "ca7a63adf41f4da33142910c967706b5c8a093350eb3e6d3aabe69a46a2872f47a39bbe77cdc1160daa450225b0e8e36f506978ce3ac9ae5f75721ef30da46b28f07242c"))
                    (acl2::hex-string-to-bytes! "b89cc12b11e3afa58908580c47b25407abbf584f8e8d4b5631e9f450464c7e53cfd7e9f9d3cf35e587a6f02957ce4c28"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1da41a0af202b079521deb6109e312c2ade48544d2b498c07e91a102dd4650ce354f3f201b3ecab8e85e21d667959b43d08f4e90fa18dca2ccca8f6ff5e9a902dc8bf5c5da"))
                    (acl2::hex-string-to-bytes! "5c297e20c307aab7f325939fd4e2883b034fd547f1dd17fb6b97ade8b148e06ebbf3ff60cbf469e4933d5f48f0166cb7"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "dece42c8849be40c78b8de6da96c2a8d7e940545b9f3f39aa1ca03ec60a85471aa84d8e29f095874f331b90a4c157da9eb048d2c8fd235399672707366c766f10bb833f02183"))
                    (acl2::hex-string-to-bytes! "bb509e33e9ffcc4c01233146226ee9364cdac5658132460a76edf617a035b197c86434ee889438581458102618769382"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "952008ebdedd480449bb96a025576c5f617bbb8307958a010767e0d736ffe5a196ea4467d8a5d3ba1f5476ff07b6410ae659dcef520a2c14e3902f8b399a289f41f5fdadb502dd"))
                    (acl2::hex-string-to-bytes! "9b63d9145bc714a8253dcdb8341b2f5714eb58b9d4b22ce45aae07f51297a3dc9c5b691a8a3cd438dc5bd18be400af21"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "100132c315bfc9c4fb93023f5d3500d7208a68acb4d2c63096232c361a161c4c67c0a74bc3e4d72c11664b1d970321d405401924b3a0f6ce2b1a2899e7caa9a55ce725fc37f55d6a"))
                    (acl2::hex-string-to-bytes! "b6ca04467ed3e623dba36f2e0248cefbe134cf555fdc14731175eaaf08e244ab0b15fca2f173a0ec98feaf359fb84a11"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "99cba4019f5478789e674e08fe5d6ceadd698b0757ca39c605457c22c3d3b8ffb797d2be8f12960f099a5606b908d47207b2636a779948282de3661bb08b1b37ee576590800a492730"))
                    (acl2::hex-string-to-bytes! "e5378c7c251ae96f0359a30b3134fd77d61d0db68c42a1a2aac293195a596df42f677e6cb98abec90d6722baac63fc86"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "bb327a0bcb2574df47080e8c0d8a45ee1c0424ae0414dc0a9b8717d9f27d8ac987c7c9ecbc946073884d1fb96dbdb583aa758186b16fa429dbf15b8d5bb48cca71469e7ce0ad8e7fa14d"))
                    (acl2::hex-string-to-bytes! "0f75e65ff8494ae28d9a0a2e65959653275fc34b2fa27b9e10faafff07c45addef3b8f25953d5a2e54e31ebe6d429d26"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "7fd9eeb5ff368040d299fd17a943b21d65deb2eccf6128d18a33eb174693538935374c32c333a867821dba08636f20022c2ce01826c7b7e41640ad186f90ed0ac647d47086744867e5c54b"))
                    (acl2::hex-string-to-bytes! "007251a2a577add048b1edc79d96c7df8fd5b5fa0d7264f122e4cb54c50bc316a8bc5f4f9dfd4469e29e9b030f563a6d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "7ca9e369e82186984d5fc729e111a7e5d8ec19c5d74e13b5ab22e4993b05c88ebba6ba72237389a6e0722e12c96c5d6a54515ab00ad80efb38665a76e831abab0fa5cf020807078441585de5"))
                    (acl2::hex-string-to-bytes! "3ee8c4184de9ceaecd0d3aea16271835f3d45c873358c93a515539c38e819414ea63b08d0a109346793d5e0f703125eb"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "5b4d945d55dea22e37821ec396476a4bfb617d2f392ad93afe67bcfda9cd9b725bc4ccdf516a83fd71dbff5a22b005fc61c58e471240bd2193ce13539730e63232f70f80308be48dab7266a1dd"))
                    (acl2::hex-string-to-bytes! "df82d242e4cdc2eb40bf3db6a56e1aa0a66e553f1914bedc65c8cc6ad9564b6e85df59f4c443cbe4e0aee05986f7d690"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e865f4a42bbbd0b73fe275b8ab90d3a9fb74ec5070192d38f60efef9564498b9adb716f31d50cf77c20ae4b2e85515307bb8d95fbeb9ad964001ac550dbc60cf213fd8a522edfaf54e5b1b93b2b2"))
                    (acl2::hex-string-to-bytes! "091fa9ae2184e2268ef9ef23c7c809efad244536e00aa9e8b3a6c228d90e31da051b40f268a13bd6f62e69c91ae8cd2d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1d408c7b68e168f41bb46f9b2e9c8b04f968e4080252546814cc1cb2917dd5690886a9600a09c2673aec0329a4daf655508b06fc1646ef3bb3a472191d964db214a96a96fa89576ce4c4f6dbf1d176"))
                    (acl2::hex-string-to-bytes! "7e23472c03431925f3b4559d886e8d5d837b3d39b8efe1b7a91e61a13810c4dbc2439634d7c6ababfc66e9b18e6541db"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "54ae030a4e27a05c1ea4a12e69c67544af9b4044cf157dc8cebbe8b2d49f9bc0779077603c90c5c55b891d3ac33b87b65e79e1b19695813718191b3bd8b7e42d5583f7cf1e60f84495b8f869f3719969"))
                    (acl2::hex-string-to-bytes! "cb65f823585773cb8802b6339182f13752a82864c898b445be5a11a969657ac2dc4a3bbeb87ac0abb232a2b124171096"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "f73cd386f73d0c6ade05771b33117117c602e52693f05b47e90032eacc39295f9793258fe6512eeab291baa0be222e143295a28e8697e42fa27ec02b44217f32a1edae2f4f35213562ca37b6d6cc5ef72d"))
                    (acl2::hex-string-to-bytes! "f665c4d17a83d65a7ff16bfce279b58558250d76af68b8eb943753e411a57ceb31c1a131e54bcb7672584416e3d5719e"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1d259612e6867e7d788c71d03c5136864ad6d84f24eaf913a34e69333116f812395288d4dcee6665e6d7dabd005ffc6327e3ca305cab78569d1107a115e619fc90110436317925066726774d1da3639c31a6"))
                    (acl2::hex-string-to-bytes! "5dcf512e2b93d6ecdf7c3304534554ea79d22392e59bbe90df21e978c9fa3b34ff82e6dcfe8fe2236aa4af4e662e2a9d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "da8734414c45fc1d5a75a3cbacadb1bfb523d6fc391f882d0db0eef21f9ffd78b6a1e14cfad09e71b65cf7b05d7e8f2f4bae4e454e16068d65465639c729cfa92738563d37edc9676b7be604ffbc68ec3b6b72"))
                    (acl2::hex-string-to-bytes! "8b328a31adf67dc7aeb864a359628410d5814a2f0cc683303f61432ce32177e1f538feead7e5000343916c7042f8b3cd"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b228c75903d80fbc6d1cf629ff1d14a92ec4bf0e121fd97bd306ed265efe7a5d5c5d8fc764af98ed6f5978f88d7cd8bcd71cbef6a58261d201de3cb15b3161287e6a104cc2cf882d839f1da0d3f68b426cf08ab9"))
                    (acl2::hex-string-to-bytes! "fc92ba4eac9a1bf120a7b6c2cc30335b9615b1a98e55d14854ff872966e71040737401c6bc08db5842ceace14cb7e7ea"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "c90d473a6fd30be9a98bf442a9ad65a697d4629c33cd517dbbed02710fa8ee991360bc8e557b0a0bf0b869e6b0c3a9457607580edec3859f2060c9c0340289d53a5d755918ca54876599045a86a9bcb8163795ea8c"))
                    (acl2::hex-string-to-bytes! "807582b2520e990cfb74367343268b9148b2519b9e7cd9182edb3db9ae7afebedfe8ca118130e2ef9d31af9081da8222"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "6df8c5c28d1728975a0b766cd7de63bbe7f48c3db3e6fd5a4b8df6e3905cef0351f3d973b4f2a7eed80b0de5b85c877353fb9e930ad2679149ad4cbe69910e68d5500b096c5abdbf27d684fcfcf1a57f02769283d5a0"))
                    (acl2::hex-string-to-bytes! "7bda57d21a4434aada6758e282e612a4c0f41b242f9c790804d5bee25b81a821dc6f2a0ba56f1b3704802c9a6e153d85"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "2cfc76f88cb6fb90927b69526ad5f03d6bd335f4f75b52b6a3c21e8f989ab0d03acb1ebe07e68a87c1b5607acf17d976e10ac4a3d30a8561d49a5e7ec720edace9f5f632b4bd63e104f4894a79caad2e1c31c736453485"))
                    (acl2::hex-string-to-bytes! "e16670ea837c259e418d3c0e1eaad4948c3457e15b1573056e24da25bff5c66b7e95d24c6bc1b8d6c2b812f64adc9553"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "3f05108c2f33d39b3aa9e73efbad4b011b4e9e9fba409b7611e7e03956b2f3e5e0aa86f68c4bfada5f9223a66d574b08f9dd797cdda8f3c32d8e01921711f4870dec676027ecc56fc2010b496e95cfbf071c820f21edf25b"))
                    (acl2::hex-string-to-bytes! "b272bab680f3ab27de72d94df384323f8555f1d17facd2588ac8648def2451f82f9b99c05ead8316fd181a2cfb97483a"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1ffbe1aff0a1e7fa3e68be31a74612a1519b59397e7007ef61fc015f316d55b57be528cebcc2f09a2f22e3c5e4a6ae9612776b17ae87cd763c1a9eabe6846c5bcb347ffc99f10e3b5e64b29a9bd71a5e9b3c01a802715de2a9"))
                    (acl2::hex-string-to-bytes! "f08bda9d6762607519d53fecb0bffbfd3ff2924854833a759d631e910c42ca86741fc2e290af42e94b94898609b91390"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "f755d6b5642378f2847893901d9aa91c54a4b7abb97c5c7184063e8f1e97aa2de4ad7ac927dd3cce770c906921e2d298f67cf9844e61bf104db803b265b86b821c5f4d901067d07b38764e3f6c95fd4f28e3cfe48d8a9694a8f3"))
                    (acl2::hex-string-to-bytes! "f85e98ea054455242280854e97c4ed399b85ee7bc5c5fc3d62910a76f3a9600c3d904c832b70b58d7d998db8dc978135"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "773577f3a642c4f13b1cb1f4103e9f6b2da86268a52f449cbb174c8349e3c0dc636ce85c373115a337eee26f7b70ba1060a79a1c76fd186399e6a5255db80f83b0be4a34ba876f7908840553ead380f3195507729d067ac2ee8eb4"))
                    (acl2::hex-string-to-bytes! "cc27869cd7e63695d19082446b068b77dde4e8604f8c0e9ce20a1b71aa9eff1460f32d5a54476275bdee8e7621491f46"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "651589927e17e1aef780690f3100a377f0179b18b31fd5b4418c84038573fc559b496a782beec3dcf6e9faf5aef676e10bbec34b1be5888fda49b91e02890d2524c5b369f8a54175f29dedf8156fff690cf186ec77104a798315033b"))
                    (acl2::hex-string-to-bytes! "da846042fb908eee5f5defd1055ff3e57103708278d379a8681f58bedc6ef89670b9f957c4e0edcaa42dfd8cd49df6ea"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "678652600eee42580f73623412e9c011cc02dec4d4cc1b79b27b6f9939695bf2185b2012ab0630f317d2e2de95dd69890e430783e99d7ed121c7c8da9ae70780b5aabf9022d1435cf5ed6da6fc6692c050c2b5f22b24fb1cf9135f9cb2"))
                    (acl2::hex-string-to-bytes! "8a6ae41c9bccc16eac4860bd5fa91205a86fbfd09692578f7f36b3c135d96f9e9001c192dbeda975f7375bd43a23ba68"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "416d3fb7b401fa5e78cd96d479d8860df147eef03adf13fce1c61131fb89cc2ebc63289745bd7db9bef14571a55318496572dbe52b9b349ef59f406cecd68909f364325380bb75f3aa62503c84f47a55aa6b9c9b199ebe414409ff3964cd"))
                    (acl2::hex-string-to-bytes! "c5f20542e0c0ac1eb433de6229fe5baccfd4502e2c2275439385efda6374a1d0fc50cd9bba4233d470ad91a3356ea315"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "6f09e876c0b83c9934ffb777f006338c5142a31375e9b21cfea9a7de12998c4ea6708ff1fdf5a8ee6bb67c675ffd8209a10064e2d758a8734eb48f07f7cf3d43b09f1bfdc5d07a52b77079f23cec28bf863bed97c859276df7f7129fce71eb"))
                    (acl2::hex-string-to-bytes! "b3c968f3025f87dbd5cd3d364bf673e62827c35889532431becd87cfbe2cc75b7ef45696d19cd3452d0e7c2b69d09544"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "0dc2b226b3431c69a76addc018fcbda22bd72c8ff01ed6549596798bd950f361c489a09c95cee2dcfd6401208ae6368d6630026b417cc4718ccb8b42e8872937de66895fd09142c42f066bf0ef3ab2b03803a8185fb65fc7148c376ddd4bf58a"))
                    (acl2::hex-string-to-bytes! "aa645a4f8f602411260ace24d381f3f5dff0000c246343eb528e3dd027cd743815737906ac5c74ea83c2755e56b99509"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "8dc71c84c8772753c86ab6afd80e8d1df9b0d7e8d69ebe67fa883a82412c26738c3399cab95573b4d3c4367c85c81852d5a6564c0fc7caaafe16c05e62aa06cc9fa542ceb35c88fb6ab82c29d5dcd530f807d3f1c3bcb3974421101d1aa6ac112d"))
                    (acl2::hex-string-to-bytes! "12239813097124e6248e7dbec985a6a25f622b1d07295cfcfbaff33b847df7fd94265e439fa535f3becbdb576922ac41"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "3df3edd9fc93be9960b5a632e2847b30b10187c8f83de5b45fcb2e3ed475569a8b2ed0784348f9dacce7b323c6b65071abd8b32d1022b1e12787bd4989d3c5ac329d576ccd7608dd336716532e9b4c7f825826fb2e343623ef85c6270619bf5e3b27"))
                    (acl2::hex-string-to-bytes! "f36590f5211a9cf84eeb0a3b2e5dc1164e813191cda7cb883f3f4a074605ce6780cf2f1a105658706fbd2829db8a2a58"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "ced0ec6500cb891e6433d104ba5b1c1ebca397f3a5eeeaa0f0561729621ea50d4ae7ff1d0827178dccd84e4ca06d9891a90adbb7a9f4994ac947cf6296e71d2f49b826d64b123a7bf86f339fa4679caddbdf19cfed7d0206aa5f527a6b0cc00f52aa2b"))
                    (acl2::hex-string-to-bytes! "c2c2d7d65d0b9108648e3233d15fc4e4cb62ed8fee9cdd18ab44b8486e2100fbe45ddcf74f46c15eb77fb1c893c12202"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "afa4a2c4fbaffe838dd149c78ea7851ea9396304b41806a093a90aae59c0c5bdb170cc9a7d22b90cbce52cc1b105108942df20c29ef3a913223b915e7ebc98ef135adefaa0f0a6441ea05920e868ce9d1ff6c8fe4dbec06a4849e5e55ad0627f9e09dfcb"))
                    (acl2::hex-string-to-bytes! "cf6ef91d8567414f5c7f0b1f4ad09a976afc7c8327a382fca90f5a136b19be33100a664390a377f8d8a3015fb882125b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "00f65a485bfd381113d6e79bf9e0d5e518c891988c073ba198ac3a20f25c2c81619723e88a3c0ed3075075fbbafb6a7b61d3bc336a5e6d6f08d166c4861e6a3bdc2e49b2806b567e7e821a55cb674a6cd699f7dc61a7054a8ff3dec73eb667f59644346be2"))
                    (acl2::hex-string-to-bytes! "809c6b5d41da7cd10df90b02b193ac7d40cf2e46c139e9dbd208a988da2b25002cdbad1db2ecc1322da20b7d054e5fe6"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b9ce382e1e82a873cc444248a3008c2cf64d18759057abe8f91c9d87f5dc83aa4eca0c51d30829b9a1d2712da1fac31f52942d77c9f20c2bf6d3751028d7d4f0d336d3dc92b27ec368caa4444b3180c1e37e98b58f25e647a9a6361f0b04cf78d17955766168"))
                    (acl2::hex-string-to-bytes! "18cd10b3ea907b3770e8eb91c974666e2da2525afe7020b872b3ec6689e5e1cd0059dd4fd49ce44d75dc4c8430c322d6"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "6778d82f3a98eecdfac55ddeebc52476a070094fbd65831801fdd60f837d80d23b90d472c5f4e5ca6273a50f40154ea8fb94013f6310ad18800433a1d379c84bdf799a99e8c7b676fbcd29cc2ed66552297de7a6e565179bb42b70d48299e0925a1d72ca2c792a"))
                    (acl2::hex-string-to-bytes! "71f08d9333df5cb885fd23d6cbb1db84f9b55908d069df50fa4795cc713a18439bcab8daca078356f5c75a619f2f8782"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "ba3a54a77d7c2b8ccec6d53136cd48827c87acdd1cd86ad1f56e862642ea2e1dcb4093f85d60f7bd77160738462e6c3fd3dd9d3a7c5cf7e2a1d60f489f84471902179f21f656ce0fff089278ea14441e04e7af71891622565f44c428044728fcc686212a32a5d809"))
                    (acl2::hex-string-to-bytes! "3cc154f0542d8e3cacf9d79f231416816178a7ef2275fb257a48a2f763ffa2e15a33c27b970a416a057925aa0412d268"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "7eec4f4f491b4eeaeb1cdbdb95e9511c2872372bf64a1f61cda1cd8032729c8beafd1edabf78036d80023c814ad8606106cb4e7f33f214c3e69c0f230e885474fd594f7f2444aa581e0c70ebf13073d89063eb81a43c5f608b2fc99fa1bcf5e2bfe62a6802e70c52ce"))
                    (acl2::hex-string-to-bytes! "2f8c5682a07438043e55f0b7759fe7eea5d9bdfc8b0f89800ebcf777bc05a941ea7f3cac45d4659de0f505d814590b6b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "f3715b9e3ddd7862e15ee87aa23f1aaa0580591e55cff3fee9b49b42aa0c0cc8cfb8efa3eb96ffb72ab06b83d7b47b3d22a5772421cfc51214005150edf532af10138ad45758add459908601eccc3703e810002a2e4c6202e98d84281475d55d3de9f3d98809cce1f665"))
                    (acl2::hex-string-to-bytes! "04e7d55b0eb4bc3d3a21cfd2941dbb4dc44706588967186b40da54902aeea97b262c97f75e37ebe3cd60a804e7b9feca"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "dfd7d792e162bf7a889109550a0fc4c415232af0c0d72dcbc2595299e1a1c2aeae549f7970e994c15e0ab02f113d740d38c32a4d8ec079cd099d37d954ab7ef2800902cdf7c7a19fb14b3c98aaf4c6ad93fe9a9bc7a61229828e55ad4d6270d1bdbca9975d450f9be91e56"))
                    (acl2::hex-string-to-bytes! "08e5ef57d0c2aa23edfc75cfae39e6bc1a43b5db08b2e27bc9823114edf760367db9cf3cd9c3779755f6d39e219b7079"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "ffbc7c47f52e69f5c667f4ed578b46ff4592048f789081f3fb39321d0aa4627a6d4f261905649410a53a301c231fa787ae55c04f615a8f84196cecf2844d23007b44edd892649fc8ed10a2e855bf23fe8afd0b9edbb33296f5a7cf89f94634d9d1a2b8cac3b7f4e546f2329b"))
                    (acl2::hex-string-to-bytes! "be10c7baf94608408a0accfbc8ce95e159d08d8ca75dd6c273f935947a7ec3463e10a58d3ceaa0b2198b0887a3a24a29"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "a39f202d866e8e96765fbb53b6772537dec043322f4a7475247036d7495c987850cef2a46218d3fab36e3bcd595c0aca5e98b9db14fa484ca8c9772dfa3839f7ac3066727a50b0d5c933d82f82f1220720e8063f08bc283f199c8a4f85c70043df4fe55e751d0d71df36fa43d8"))
                    (acl2::hex-string-to-bytes! "3b2664ccb555a1b1f3ec996860146ea75ef7f3bd62028a19c26f63339399f4275a07f3c064d34766ebe8e4dd532f6629"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b2c8261853e2218dfa135cb5387810352b8962e94e9fdc8695b41e7dba6ad122d14fdd0d2360dcc039ccce8b37fa0ead6ccc85bc26261d47cbaa78b925c6e380fef1856fed31dc616fe16b2039b1ac85cdee4ce04c0497998b41321868db08e35f358606585e0bb8c3da9a3be7a6"))
                    (acl2::hex-string-to-bytes! "45b286f49fd05c45c921b7bfdbe2cb024441c372e07394dcccae0de834cd541f13a79dbb3e5078896e88438542bd2f12"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "a04f390a9cc2effad05db80d9076a8d4b6cc8bba97b27b423670b290b8e69c2b187230011c1481ac88d090f39154659494db5e410851c6e8b2b8a93717cae76037e0881978124fe7e1a0929d8891491f4e99646cc94062dc82411fa66130eda46560e75b98048236439465125e737b"))
                    (acl2::hex-string-to-bytes! "e7089d72945cef851e689b4409cfb63d135f0b5cdfb0dac6c3a292dd70371ab4b79da1997d7992906ac7213502662920"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "f419494c3c6d0727b3395a483a2167182a7252f4fd099c2d4b71b053f94bb8b3adf3b51e8460cfec084ce9415c95798fbae4975c208c544645b54c44d2b97f2ecfce5c805be61f5ba1d35dcc07afdd51a87baa990506668cf710e18be9b0ebf943f366fa29c69f7a6616de72a3353b66"))
                    (acl2::hex-string-to-bytes! "aead8688c58c6ba4e9cadb4756b465dce0fb06f1cfaa478197f2ea89414e47e9572034adfed160703c79b82b3fd7ab78"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "aaf7584d53006cbf2d2040e51b7feebd2bbf1e9f6d817cd8062a6a9680e7f10464eefeb50b07cb46b14b9b3fcb2caa3b9ab664490115d5919456613bf172b58c5388fd52646a5783535b88212717ef605314b70b8a085024d4ab1fcbe2be74609e4cbdec0730fabd3cd77151d647a3767b"))
                    (acl2::hex-string-to-bytes! "e6e79d8c61d0ea9fc70dd4dec1fa432849c3396e717b170badbd87a4c7974efc590ab8c1183a6232beff14534f004b02"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "a467f77369730201f2812204fd63ad0d2757be580d937dfeb221a06b21ed3213531d936152a0c1f09f0ad5fed19fd11e80ad982c61203e86b2508279d91d99fa483e2e97a3d6a6ad2548a8da404dddb58344f4bdc1c9ea907088885e4f532d9c4c73cdfda43c3a9e4ce5a2809096593cfac1"))
                    (acl2::hex-string-to-bytes! "5a7508c2cc096bf65a4d4d337aea22008edb9a3bae869f94e09fb526a52c3368e9b28576fb950f078b7e43b5562120e6"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "01abc90e9180fc9bb8ea67a405073ed6848bf33048076566476c55836bcb19d3e55e9400c5cc657bc7a95f1d703c390f5a8687e7cd7fe9138ea3837bfcadad6258a3eb8d65121fa8319bfde532aec0e694961bddd2b673f284124be5781100f40381b6ff99db92ea9cc82a4372e53924dac398"))
                    (acl2::hex-string-to-bytes! "96b209d7cb2c2033b38350744767fa5c253e1bfdb99fe8418bff83804df02248140fe3b77b0bfd4a79b51f70405a434b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b555d99056362bfc2bac2a1bbb71ba112d644e50b82b015e5a1ce3d9cd5e90b8b74b08d32119baa62abae251fc0015e400051ada4ecafce3681e5de727c20d47f5cadc663d46ac682022ca396a4b7ed1c413e0b72bd7eec4a0dfdc2a2185abb5d99afd50940528ca75ad89daeb9a1b61e15747f0"))
                    (acl2::hex-string-to-bytes! "abd39f79d72c6cd2e8e130f3603032fe3cef4177c3563930096df1f10b87e79cd4e2059cf1b9f8252184bb26f659a5da"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "14fb01ae9d6015ecb3e56d6ecdfa4bc0533186adf8457f5e4a5c57c687895f3db395d06ae7ffbd67ec41452009550dfc1878eec0df2eeab09e8665f7e59f9148a86b2bc695b36521a55b2302f2e869aac83f14d6feafc9e587322c3c44f052ea1c0578884f84f56307bc6dde31ba48118a0f62b6fd"))
                    (acl2::hex-string-to-bytes! "fc9b9a95a8ce1c157720cb63101a7594df24f4cc74baf735b0ccf6abb925478ad507cd048d30cde1c788806f43ed3a81"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "11ae0cbfee7bb3df90ce585f09b9cf8ff5bea69a68eeb6c22553f8ed118c9a61e7b752cc2c94f38766e63e1b891dfa05b23347b617d42fde3ec17eea6e76d3cb640bf8fa46586fb9dd5c4d9bfee04c4649571b781709f848ad7081afb6e2c746f071a551251050fd5df72ee65248ecdc24f2cbe74ed5"))
                    (acl2::hex-string-to-bytes! "3214b5feec925059149fa852e3ae285a6eb377df926504e2f824572a3aebd2050a20144e7bede7e7fe238ee83e69f72c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "a5c4a47a04f4714269d5d922ba4694060aa2df49193720c819fac93bb8787ec55a107ac9a6602f0045fd2cc8e66744bf863ced91eeabe60e7d2c1d80276ecf3bbe91f1757096cf589214f3569c2c48bd74be7f8befddb28395814780a47c180a58b0d0276a7e9873d682f473e27de7275c925ede23b6cc"))
                    (acl2::hex-string-to-bytes! "6bd9e1303579d815f58e8c6c9855850133251778a632f7b312c4b3163b29b5ef6cb9511a08a31a237d9a7604afbfa056"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "75264774af69ec7ee3125e205bd6d1cb8fdb22f7ea6dbe72d1f2c0f7e2205902796d75e379c79b11498615c21c9f52b8761a885eecc69d132b2b48c63bc074c3055ee5cc13f51d6c987e8188b030b837e8f754d40122b451f15b28cd2bdd576920e1de5806593a36d8e1e89b9ef3caefee5acd80b3e9c9d1"))
                    (acl2::hex-string-to-bytes! "ffa9e4e856d06227c1ccb959be558309cc10633173f4b66ceb382923b52b9150acfb08a7393500477a6ee4425a827e76"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "791a36d748695e62db5003a8ad367df1f051c1ac6a21d711823e8e069b546e3fa06ceeaae06de70a1de249e1dffd0d940edc6acac00c4c15504c02d4b0933658005423455f00023b01cdc5b681b6083379c24595518a47c654f5e1a10947df10c05a3d716b2a973faf98e1ee3b675816598bb8d4c2a3b06eb7"))
                    (acl2::hex-string-to-bytes! "f8d33369680524ed6c6a716d4c502de3eac891f340f40e82e37501be1a90fb61d26e5daafe92bc6ef5ec0ae431168842"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "74cd86bbed14d895301d8a54b2956b1c5cd1451eebab620b978d4ecef2cbdf7a14367381f5ee79281a773337740fb9f5853f4253c4b19f684341081d8f561b2ad773224151099588da90e04dddd565f67596c536d64c5b87e9480ad43601397507ad1b61ca0e349fb88f19feb48f770676fd562ee8259f50d0c9"))
                    (acl2::hex-string-to-bytes! "73ee8d29c30821dcdfa4441639f037fb6ba3a9ca596dc43428043785756608f6207d80b7f78e573174fb9dfd42f0b8cd"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "46612e1a4d016c41700a3b5ccb38323353bd8da5b9942c9c9212df40b4bee06be362a15dad62c8b2924d789168b2d325fe35bd5100e1e29f1ac0fa7a60a94c9eee5a70cccbc75ac214b1946a5679cb523b378d5c690751b7a7a3b80d413712feae7024ce71d6295a3d5d16515c3622a052eb862ebdab81ca7fe3a0"))
                    (acl2::hex-string-to-bytes! "cac13a3784225be03d526f9abc1eb50a762e72c0e01172a15d578801089e5c9f26e53cc00ff755909453e2964d7df838"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "1a857c1f105d068ceab0b1e12494890ec196362a48b0200a0d75d712b18fb14bec6bb5b68a33b7e0b4fdc5b77142c29c6d9151b9f884f59551f476e5256986a653d4a468f28141ed954721f2cd02054df0438738194545ed70234173ac4988b7d62812de4f2feada14f68e3b41bc9948d4139f4fb4da5939f26c0261"))
                    (acl2::hex-string-to-bytes! "b75d92b5cd3772846f7b68064a3567ae43e5010099f2b649f35087e7592aeec176646fc8c2629288944261cd35b5fcba"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "9d36818d0c5a008be7904d1917aa3eccb5ef4f38cecb8c4e63c4b2e9b4b091a3bf25b4ed0332445f894c2a4c258b749afa17fad03cdd41713a869f899ba9a085e73fa9474a58db7a950d3a2386b60f79495d8bf73e72acaffdbf65e1989ff9cc206ba8c46a368d8512bac7c7c191d713aca949d45df297b1b6594a1a33"))
                    (acl2::hex-string-to-bytes! "a88da844579f3e3725d00daa8e1e287da4bfbb2d931ebe8d3b16021154be503341d6371d382ada744f86f5bbb56cdcfa"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "4ae50ed626ee60dc5ea556e3ce0d3c18b9e6225b5620814e8b9621acf8f939dd370ad9c7620d85e6d67a229f37f517d1b580acae8df0b5c9d29d756f6d5ebd3b63b554e556469b5b4f8e7113bad1559fb254ca827fcd00425d18b0be7f2b48c2544c48d90982ec624f490be65e893fa93ac9467f35a0a8e1b56d9a403748"))
                    (acl2::hex-string-to-bytes! "f8cd943f429333c7c8d30a8576827f92c92ae18e0dbcae770601b79687bccf8c23e9e589dfeb45c3b9bcafdd545e45e7"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "dbed7612448d46cbe0a384d1c93233f02ffd1c984ba765299518656d3723b766c1658d4b1e7047cdc729459e366ef9349efc40cbd990f2a9a24db7a5045e1dea12dce8f9d9f2aaed933f93031e7b8959ac5e7bf6bbbdf30b48f7eb783f8fe292371a2f245c5c94b4acae160767a20ce7c0ea7723d97691d8eedda9efd1fe2d"))
                    (acl2::hex-string-to-bytes! "fb531a1ed181c732311e56f4b56ed91dcacc0dd6bf1eb4a44be6f87dd7cb1ef9dfb0310f4a79eaaa3f32bf3914d8624e"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "3bf52cc5ee86b9a0190f390a5c0366a560b557000dbe5115fd9ee11630a62769011575f15881198f227876e8fe685a6939bc8b89fd48a34ec5e71e131462b2886794dffa68ccc6d564733e67ffef25e627c6f4b5460796e3bce67bf58ca6e8e555bc916a8531697ac948b90dc8616f25101db90b50c3d3dbc9e21e42ff387187"))
                    (acl2::hex-string-to-bytes! "12b6cb35eda92ee37356ddee77781a17b3d90e563824a984faffc6fdd1693bd7626039635563cfc3b9a2b00f9c65eefd"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "62c6a169b9be02b3d7b471a964fc0bcc72b480d26aecb2ed460b7f50016ddaf04c51218783f3aadfdff5a04ded030d7b3fb7376b61ba30b90e2da921a4470740d63fb99fa16cc8ed81abaf8ce4016e50df81da832070372c24a80890aa3a26fa675710b8fb718266249d496f313c55d0bada101f8f56eeccee4345a8f98f60a36662cfda794900d12f9414fcbdfdeb85388a814996b47e24d5c8086e7a8edcc53d299d0d033e6bb60c58b83d6e8b57f6c258d6081dd10eb942fdf8ec157ec3e75371235a8196eb9d22b1de3a2d30c2abbe0db7650cf6c7159bacbe29b3a93c92100508"))
                    (acl2::hex-string-to-bytes! "0730e184e7795575569f87030260bb8e54498e0e5d096b18285e988d245b6f3486d1f2447d5f85bcbe59d5689fc49425"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "ffd67890ff77f344ad4f067df2f4ff1db8f541c7a2bd9ae9fabad0faebbf7d00f0a71d568c3c66ac3c57d84faa4894ab2377710e4b4c4dae0f4da1eedc8658dd0e2ee2ffac8784515206f2876eb41f98afd4547cbbc68034212bcf0c8e4a7d1d43b3ed15c621f53bd8a57cada80148ec4652119b5af3da84169d81dc69d394c8767d662044d36272b77ca04abff7b6b0cf3bd1f3919a04a5d8ebdfe7d6e844e78fd576a68d6373ffd5d384e51b5e12ec32d5bb0ac685a59f4d5d12b43b533580750345310999cfe91cf2500624fe03a65769f86a627a667b5f3b42cb01da109e124ffa48203f1f3873202d35429f32e8263eaf9bce42ef40f5cc96b591467d464d00bd743a1b0af4c1a743fbdd0846b9879e092371a5e7f6f65937f9515e23820e60b83bbff73926f0cdb9df5d02e82262cf2e8cb26af6a64c2a4d1fabecab593db510281799"))
                    (acl2::hex-string-to-bytes! "1396fea95ce0c1c1c224b50a07dd7197f1d62b993c7fe9e1cc1a56101920d4b0fecef587fbcd56b854c8c9da95132f02"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "be105a6c585a766aa1f290b632219ff864ea6674b5e3f9846d447d46813e2f92b78ea82f0b515f46511a6f161d429aeae07f8e4853b8686c191877f5a06242dd4884ccaec76e16f3cc24e6ede212c78897a1518dde07aa19b3634d4efd090a48b81a4e535359a55b57397da44a3b2dbbd37636ac1f77c875824c88bb62dc90bc517ab78579b913643d81bda11d62d469da29c50bdbb1c967d0fcafa29582db1f59c98fabff3669f8c4232d4d2332c57bf6e08613dd5db5d6e39b4a6d5fa4f35b19325c2fae79aefe36485610235007da6cc3022cec2295ac0550e18388eae156d9dba8784e2aea5ed917be53e767a26c87fcc0bfcfa870d07b43fd4cd8fbacfae1ecfaeea7f12600f6b9ef7c351d9f1b8eb048324f984e2a909d253017805c2d788edbf9807469ccd45571f5d61a05cec80a23ef349a37a28b86e2970c20fad9e7e8d201a35ebbaabd14ca249207dc6d7e2fd85c46553420b325bbe980848de57724676ed6552482f71e8473308c2ddff94aefe34c724c8c52a3388e3b541d396d6722a8e201ad3ccb9a26497a50ff0e7e81f1eb109888ceae27e1ef0537b3bd14dc8c178f0c5dc081"))
                    (acl2::hex-string-to-bytes! "39cf9206d78cfb437000b100661753181b5f262b539c208b442e7a022cd3234c58751621ff0ee9e9d3607174e36aaa88"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b41ef4a2b31d3a47a5e8386925fb90fbe980ca2cf6ba34e814a3eca280d43b51e2a624b87e973debaeefbe0f90a3d861fc79487ab96d2bd1f0531481e0bf5c4cd422b9e05235125a969b92093fc2fc74470a3e9e382948889dce1ef8b7842c7972d9dba397c0619c4335dddf981fa33cb27ad420ddb533170559338f6eec55dfa515b088b601d1ba9f04c61a058476c2261334890fdb93850e3f7bcf6bbf7c86531c631715bd0bbb91527641ecb71c0d891b8361356110dba2c61d3f17c54c7765381ad81dbfb7a7006515dcb7813e25c5dbec2eb9105320a8384268dac56ab9c4aefd5436ce7c87d687ca73cdaa8413c09b957a02b0045c08db11d01d89815456c93612c9b10262fafdbd8f96fa95de64a3a6cc7809cb98aaf8b30146d7375105d0ea9a3c31f240541217b77a391a8e973a48ad83e24edb76d64f83bf7834b7b80a4475f916f429cc0d0074e12eba2c43a9a2a48b05382d54bf0edcee34218e57c0692a6628af264a359f4d33a21ebf4f39de39f8382f962cf918963dd2a6dfc93b8441ed2ecc3eb6d41dcad85dee4fa32f21f43244db1fe6cd9438131c6ab1f8c0bb43f5f2f27af830a1733927d1e527d6965ea1f1e6cf6a83aa4bd7d8160ccb9a36b7e53acae0f89154d1bcc86f87c37562feae06f5afa0638456f3cf51f82a0f1e2527cffc7b9f415e6bca0771405f53b05cbf4db3c570b547e1cec9c47efd699b1522792f50d387e6c9"))
                    (acl2::hex-string-to-bytes! "0f8ff0eee8ffdefb8b3151b7664ce9dcaaaf3f7af0b1e290ee49ba211b67863f63fa71202a534c0c42444a52bebf6c62"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "52a6741b9ea02569dabf96ec0b3e2462866eea4a635bd7a42fad04f8e1cc45990f48ff2b627739634860f50ee8d7c0555bea883f197cae77b1305c50be891906164f9a474434dbf139b76f1e3e49155e2dae8b43e1a7a723ebcea31986c1b94dce82ef752c1e359c158dad22f4da4a0564849c5ea323ea2f434e825b23bd4df69eaba6bf027e6ceb44aa5625f2d387d2683263994739a023237aa01f7cc5fe8af090672dae9cab672ed6132c207cf64ea28087fbb531f188f218a66c7920f3902bfbd9c619c018963bd98c41661cbc007838b18dd65304e418f13b4b02185690b6b2a4922932db02d47a686bacfc04705320885690e602c2fdcd08c3fff34c12fb7d7199c7391d335f8250b3da65f4279c02b55ecc3412a70974b4dc58faff9f6126e8e4dc1f06b046baeb1fd89db4acb9a4dfad73835d1da4ade89ef67597be5073b07a4e2abd01efe7e2ba8eec6d8714ceaf4c6b1c0bab36414c12bfac3bf8302e726307a36baec4d4dc89e49c4741134acc429e4743da27f9873f9ce614d46a1dd5a8edd2373f2b488296c3f4696120dfe097f24c2d30df9b422e756d4ada64098250eafe4a3d8e0a7ce8113bfc6fa25490535172f237c6ef24e53387049287b92063e570cb3054e37187b8ba27f40172867fa202ba7744bfd66ebc7d6a853bf758eac571e39ee59b097c9eb91377a4016a23cd44ac77c2d1ffd1fa97a63ae989d2eb7447ba11a9faedcead7fe780f5c9e345b980f1bfa6ee4814d41cdd39f7383571d93a771008c8c585cb56cc975b919aef50feb0d75ade594131563a0bcfcc2981dc9443246acc38aae284ea453e233728790512484469d95ffb8be237d70d5825659b45d337cfef43d95268"))
                    (acl2::hex-string-to-bytes! "45d0bc363c4e0d5c1993570d7366847aae64ead4138d93acd843f28d3f719b02871736ba4905532e27f4bc8b5c2ced85"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "cc990092fb16b6311fe4d66dd5c472215e807f82e71d34f60e58b32aa351eb5eab81bd04abf15d8a23aefc745878a4ad56ba631f36bfab819d22f9708586b5c240885917b9f2d6b0fe313485fb0a78918888ca71a739d72d069a7556cccf855192971ef2ae80e84db857c5c847e3b775fcd3fe9fc2f480a353a76e01ff9325b3a99752be12025a065a07f966c36b2900b98145d20f586e8b8c332e9ca6395b918d6e3ea26be2f585ea291687f80a95dbc77be83217ce688cb725fb4d492bf09347062ac74387817273a465526c29c4ae2eaa2ee661bc384ede651e3c71d0ac60dd455e542159c674fa622ef0a92e7f20f6f56117758ba47a08dadf93a59056a15a1521728ec08b34894ab98937838cd92843c7ecdca475b83e6e85776ed3b8fdc35dbd4911249a5a27444d449c8eb382cc7eaa8e1d6d72af87085a125301981d2205780032edb98ddd38334962b9bec3e88d5024c0506508eb90a48cdc82bfa87ef49bc07983b175b20eb4c1106b7343865802d74eb4146a72efac0d91bff547e0146db06f4f49bbaeae16d42f0b9cabadc707cf4e01497e250bbabcfb1e01a24f4c38f99bb74b07d901a1a94603ffa266c2a933489ecaec0bcd15c1a108ad60d38442c285247d003ced9f69ddba65d1b8312e2bc4f55f5b2c1d07450be6bf2a74a4d534b6ffa94fde5b563b7d9895dfabfcdfe14b6dd27391e4d670c21e00a73b953777b5ae959cc7340f48ac055975470b047a646eeadd4108cb694d1a38802f3db0b20a9b0a8e005e12b7f569773b9546e52198606e4144732e5934402f5c59858a0d3e2ccfcd90c3a7b1dd2cb8d89c96c9fc6eac2746a30b476700faace4bf550bb1d8cbae027f4a165eeeb6e4f564e1fa336eb8dd9565ba5dbc38af79c06b967dc660df6dd5d6daa6e5af70c339f611beef20ed57d69267ba7f64c65c97652e26735fd0d5ed9c5da16fa8c01c4d590c5e580db35fb033faa4d90da91e11b3074fe15c05aef3d9ba4d23620fb9d895bc"))
                    (acl2::hex-string-to-bytes! "2d24f6afeef2af38849f29ebeae7012717d50f5a31561ebb9e5760b83766030d1d2394c8eb71e030226ecf0159efda87"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "546a582d46de5b45d3f71166ef69aa0704848c0a3963b194f4674a1d9de4a9c3a6195de4c8669feb8fb9ed6a184af46a1a88b3ff2352de1f246b80b33028e5ef816cb7dfbcd667437539ce9df64d15883ca4d24b43df73ae8421e00dbbc45caaed4bac47aa987cf59060fe2710dbd40a75c0301921703aec1a71b250c61fd69df39e86441a13a6a8e514c42fee5dfa98cac4168f466ff97ef0443da7daa11b709a4d8bfd95afac047e3e67dd19394fd94598ccd53942790f0e80ca387eb2f2caeb41049ba2d458b362c7305a18d99b76ed249a14825382457a0fa12b3aba2453ec603845caea705a999567ce73a3a616ca51c874aa08dcc83ed70aa996217ab4a79371249459bda42c925461b3013862e5059d1c38cb4e4162c24dda1b4fb99524c3f560888a073622a43e395f5a180245296235b403a158ecb7b1e550adac9d6ad60e202ee316bb83b56f81ff150f9df2a660e6a7179db4e69d7ff2bda11a7581b6018c0fcdb1d957ac2d5f1c9ce5dbf562dc5898710afe575e002beb94cac5714919ceb2b1a238a771c1fb3d566266554721b20f78b0dd48e179b661ef853442799d0b9ff5af817aec737eeaab4de31a3c761d949457fac00e35bba718136afd2cd46e5f03a15cd9058a10ab0370fd4d2822c1b02dc6fcc7cd7b6346793c10aa4c071c79413f70cba450d809c6915af5f56ec5eaabc68c650692d5c578c8ce4fc07f407519536658b10bff8b5c79a21a8aa252cb771f52a3a1c33aac7f7e4bf6708100dcf2a3e5881404acba2ef07ec4cf068fdaf799f5888656ba84f52730a64458843533f99664b583081ca2be96fcace1a59f94fb1e1d9e937cd4428b2468244bf508461b21f37ccf3b3b21aa91c9097dbba2b60f7469cfc1a924cd72d043f8a17bde92d12d3e52326c9f147d1734ce20fa6cc69ba074022a68fd00a4db3fa8a33d3d7fe5624b1df2c4cee1ffd244e3138a842a6849462014e2e542e8ee641c9964b0751d6a24c7586a52f2f86701309e92724c3e18e80fc83f66fc08b1bb3dcb17bcdc43ead136ac86b3ad3081739e083abf93e5b561b98ea60894cacb4bf1807feb1c1d8d58543642777c6fc6a3139bacf26624c7b904ad3f5273f7e51c2101c4c49d74f8c3073dc428c67cde98789d0d93"))
                    (acl2::hex-string-to-bytes! "8103d79f14895bd39e568b4c8cec9a26f03966763c3b2a65174a5d2829ecae097a7c6971dfe182fd2a7a9ea6e329b6fa"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b8727e619a77b72152a46d80efe1c1ad34f131b83e8f614967bd1b926b599482c9f1f13b6cd809c78b796d06ceeca6ebb2c7bde82e3701a7974f04fbc19fe7166709d7fcccd0d838a3f2c1cd2cb8886fa18bc0f3d00960050958c6193abafca45962398b0287abdb62b673bc536cf2a0c8e443c06c2e08c986816ad220d25b9268a51ed056240a1a8f6e8bba8a3169e6f17b4d374acd815dd672f710b90e8d495c7a8fb310cf7d5cf4c10fb1182f77a9e2f7e640b10f5127c1d275ba4f971932fd88931c77b28d0a57580988468da351640216c3eb1c02910f49edb8289587b4127832f3d4f2c18a81da80275c7475463f41e6a83e1b1c01878166f28c2f603665f174206fbaa2453af07d76a20b76f13b5e5dd7f33425e1f21fe84fc58d2a22bc9c628b9030b7a34942075c5f7b2db8200184395191a79818ce7bba1ad870dfe021031d7f7c880d787726c4ddeb2304b5d161b4a257298ec1342818d238b7d5992760bc83c79de98d38e2936284de3ec377ff312cb1342bb5f9f65236d4896cae567f51e6397fa0512c44b2376de8cc129a639939c54fd7bbc6ee508ed56d0ce9586ac5eef9b59b8cfc5309a4cfc9186ae397d20fbb02fa9d4316fa2262ec1d10d229fee430c741064f1880f0d2078893faa2c6f6bcd5fc2117783839a48d07ec6fc2e610053ef986ce9900659d2c8654868c5b38f0514510ea1cf31cc3d83245d43f64b9a1a02dd621e06f4959c07977305d7b1c7efd3d3166081114e65507e3aca75f282b7844bceff5b7e52d49c9c29f1e0de98e614d4e31467c67358dec8d9d03c1a6cc4ba601e0debe814251ddfe7bd7f9e84a06db2df3de1a5dd9074a0d4f5a7179aca3ba6d9da56e65c05f1ff36e8e2761d873519755d6d540fa4b8a296cb457e9f207eac7ce52add8cefcf6681e4d7f9b60301554fe389456bfe212aa991a5b40c7009ed91beb652394aeca26331be8e00bc6912c980c3fe951534c57809265c452153c692a7a7d8f3b0004f656014d7600bc2be7c858bd3fd8269efac9275dbfbcce2d40a606dd62eb770a9e2f0593a6689a4d4c3826f8a07be472591cde2b772cb3d92635986de4cb9d45e66532a3b86be95e25907dd75662142f3e1fc1b735cb4da8e27f37155f04f91d98a18d65f43030c638776642a8b63e3d6f7dd7e940c023033e2debbd19bd13fb7a8a20c1a2e0f7ec4e96e721dd9fea04b4dc28aae234bd5000e7bc4d5b1f744556c9f2a009a2c73a2e950b5c3e44a1ecd7e5bde0614fa4f8f154b3ad6d52a365f2050be8729f2617"))
                    (acl2::hex-string-to-bytes! "2ac3ab5299ffbac1c27300859a92a69a80f510a1f28b7054f131b43af74a0df5e298a9eab77e49ca81a578c309f5aec4"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "70a597d1f470d69d7b7d495531c3182704dd60db5c73429fd7108c8ee22d86ed5822adfe6352f28f93023f46cc8d60bb"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "81a031db32d465202a2aba19ae6072148f3fcf6fe33fc8a843c0691cf845f9cb7bbeaaaa04ce2f1476bece682632e20f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "98afae971df98e02fc3f503d0b12176c2ed77f4e7962de7194c7caa2646cdba8e763b13260f7390db5852d9a158eda80dd09269ebdfed1441750fba2a7dbe78069befe30b7fa17c1b23df576e4f8117624b79ac5b9b2bb73c29f1c14d5cbab70e6699c7b6a2aae87bd0eb1b8c17945b8c989e4adb5a723685e5c34df3466b0de250c93f9c1ae89c59ca06848f439fdad94260b3cf804918ab22f4ced0dd47d33ef4eec9404af2de4189c220ec55fc1bd4f37175cccf28d4188be6b25a73d9ad80cd7ae327a54244df9aae7f17c131950c06311f47799a0f5a6b4996f4cc890334450e1bd6cc6f5670771c0dc607f8eceb15300ec4220510ed5b7deb3429de6ae5f618e1c222c28990a9ab4b4bac8a7a0e8afec2c75b029f4dd6f43d9d2e524b024b1bc37e4af454dc9cccc62d219ed2c2d1ede9b5870af108686cfca44c8d3a05ce973f6fa77d5ea0db00fcef098e7db4a48dc2333896bc898a18b6ce06d7ae8cfb0bb41abab60d89eee85e05c6d1d1505d6f1ab4d22e585da6b7da8eefda1ef45af9d0041746c7e442d36815dc117432870221cef64a6b01589af67b59fd01d868e201b02f98346c7152b973dbc5ecb79bd795c617735af11a6298f7e7c90da59f44d7a37632c97ee4f6b0b42857917c166fd54eb821aec9668f10991bb688e88892a0627d9f0d98283fec6b545f08dc712b6ecd5addecd6c7711be3686e1a1d9f6c2a51c8455754c58edafde234c97556aad0d662208bc5a7c9195b502817c15e31a405c8be5d111773fad0448139c3116e99de70620202a1f77755559d38504f73cc0301728569c88fcaa70dbd00424bd7285a084ae61a207cd485fb35c256ddb2f0be050a56cb4322c8d6b050836f60751aed9c13be78bdd9bf51b37a6c41937fb0bb45de47c201dd83e3bee459c6f9e8dffdae84e19ac48e764d10e38cbd1b2a9a6f4676deca9f208a428bd0e9d57272e5fea1f7061cff1bb7dc95f9aa05bf43fc6e795c9261759c2a97c9beca1754d3db8538e2826ba7f1e284a5355ad7fa6c852d6107f9032f910347daf661092b5c1f15b5ffed1369b194d9e12f0590770493209ff1794c17bf92e75220453760ac2f24bb77ac80bedb2d40bc3f30bb9f9cc4ec88b19998e86867daa1b4ed0e9f249a36bc12574011cf3a5da8fe62d5fea588e5bdc0e075fc3bd0a65c3cdf0b3155cec02a2863de0d1e4a8539d5226bfeaf6da09197e45fddaa8bded41081b9b1d06b55a85268c6fb38cc6bd16bf5ea2a2ad1a1c6befbd30724f7e8cbe87ba6baf4ce6b9908bc7776bf3e56221b33dcfb98536daf9ebd6bf33b8368862d333e976386768eb3f58c0a021ee88e7b82b10d886168cecf9f874c7edcb2a5473f0de17d120afbfcaa512fb1fca788f55902ba7b9a2796ba28b15d3f627babdef57113d1ff9d534b2183524f45f7a4213d4e9cd25c3eb9f6cf76b02d93080c4d0d24c8e2bdb24718117f0c2a897d528785f12f3d25f28afe1a53ef9a0d0c3929dc855b2fcace29fbe0c0d7d4459d3680e996ec0c95f79c489fb566353eecd4edea1f5529a5658e5f25e52ffce8cba2b5debc4dd11ac2606eb627ae6f872e4211cc4f4160ebfce6ca47588ed511ad398ecbbdef1b22c779c8f43742a99435133dd9b1aa55fd18b5b5cb977f2d2ea66d8437282a5b5c9b4affd040ad83d4b5cabac8d2fa89572d9dba392e6b45be4bb12ea7979"))
                    (acl2::hex-string-to-bytes! "b31950d784b30bb1743918b99fbd6693b30fde41b0b9c2e154425f048827a161dbd4f3b163b5ea8b8d91bcdacf22e84f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "e35b64497f34c095a1135e65b6d417d69e8587fcb53cd700d10cb8a1c0ffdd97144b6cd1221b25169ad42d1826d10bac"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "5a39dfdea913fcbcc5549e23b310c7ad76f392fdbea1ac188d529062b9961a2e32cef378f951c4d29eac03b3fed8e662"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "c4d5e18acd6e425d4a1f03c23d7a48ccf313c4f987b9a1f4b4e3688c5ee4d7a97a3dd31fca20e8cf159e1feeb0df4e65"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "fbaea1239fb68417d5fc143c0a2386ea75d5ef93affb80108e6faeda4eaab7b30ad620538ab0589a91af34b78f06dcdd"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "916864ec799cc0ccac4b6a806a8c02ac72fa2c3c9abce6231ae7a240d9f2f26b819a711de5480aa334638cae55cbe6f5"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "5077b3d65d16446eeb06f10c73ee1128e29060ab1ca255c6cfec53af04edc2a665bc5dafa937164bed9d01d0316f0ffa"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "8f4d27faba49fe0bffa92f430faccd1990f9b745442d455a5c64eb39edbe7c9d95749a644ca8eeab63ad183cbc739c12"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "cb7a6f63e6186009b6857157690ead10a5323917ef6035d41bc71c224b5bdb0ddacbdc2a59e6ad2b3731678b1848ae66"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "082f8540fbf6774f3858ec103c57091ce84b1479a791f87495c44b3e85518993267eb2ac0cb44e089567c562aba69709"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "34136970bcf300d0b2553a049fb7d598acf9febaf3b79ff79264bf9a4b0b27474b6fc7c27c5b19f7242b7b8f594da916"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "a6f88871b9f52e19ce23dd00fe0162dd6a6bb39d66cdaf2ff84b55aa9ba5aec9d2d352b374097cb35f0e1b71867a3f6e"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "240602789ea8f91535fed9ce15b79d6d3e6cb54cdd4cb55326ab03ea76e0f89d42202ef229a55351b13a1ec143c29711"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "8e2dff705dffa8faab0e5e06375d8a188602a9084843866e4c991a450de4c4503ecbd3667ebe432b353df3181f9d9726"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "fa6f2bce66e628ee4638b8658e0cd36ca630c71c7e1d6da2189b4e8965a73573fdd137d23481ed07a8405c04b1c9e48b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "88a7ebe36ed445be23583ee13431e6291a08ce9a42b7ec012cad0fe205bfa04d84303e571f0df0803c76d7fca5a7d8e6"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4e7fae296c6d80c6508379f94e01d80cf3bf9e057f579ee91df6d0f7c1fb85e4fac45194b91d0b1a9bcb756bff848a30"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "8a981033d7b4eb4cc9648ba26ad9679c1550dcf32df17fb80ef6b299440f7560fc057fbcb79e8e4446c2fbb9f2db918bfb829f9f44574c504c302bb460770f200fcd21eedff32f7d3eeb477abb5a855e76312938d58cecad79db953d713fc7c8ad371d50b57b79009b28986d2d1eea1d70a6a1464a5908d77f3d17bcdf30445ebd8a9b6aa2fe11d443c1161bb1ee69ced02a304f0a8448d7449edaba11122fed260fc5c50f43a20120fee80bad94aa7b7e03fedd7dd553b4b24805bf71ad2f7ad20ec1e86213d290182637c58bd6542b617932770b16d276b082c15166c8ed0fa8ceb75b0536498815832ad2b17d0fe763d6f5a61c0d6c26259336fa582d10b39099d0a75e74b0f1fe5beab9895185b66b606ef01883cacbfbc48b41afa2416f72b53353d795afd4bbf79e0279195ba06261c6e06741946d195de9aed4f0284a366b0ae7d660ed38339f6f36eb9f3cad2e1f24fb710c711467a677ca2ddf6c4cc5037aecc59e1325509b85dc46a9cee6b8e3b1cb9faf09c9cde65e1c2336b67d05c5ade90b17c8a3f932077cf200b952e8f494d757f005933367c52d1fbfade6dff9cb661093b5bb16ad1d9f576ca6f9a79cf6a702990d7b816fd5838740fa8e3be6fbeb3cda82749054f1ccddb43de4db21ffc40970c8889f47b5a8c482447d02d56601f7b3d6d74d6fa88beca2251a6dccd6b5d23edfc275378f243f4d34bc1a0e342fa9e859ea36466d64ecfc07a346e5a402c7caf931ec86887d5f6872526dd98cfeba3fb4b7332166e0458b8a884cb490344fdebfdeea7eca3d8b9da547e27e96f592818f462d47bb865f1757d06980c3d2d140d9d058657deb2db275a460447c9c9976919d0a0195dea268158384936564b7b9ee0f1f84c810ac708c70d175637b38e563c6b4971c7177456daa85339d726811a6a42523d433709f8fb31c57751bd597adc9115177c64477074c53a993854c10ab96548ac9dce10729752933da03da5516ac0b76cf6672c78a71d83d7024cb7ea2ed9300ce94676f00bfe050babbade5467989504bd76afa9f21fb365cd84d2236b2c9ce0be16aa1bc9829f83cccb80fe638722514a393dbed65a4de7e283ac50dece03da94e52630609ec7188bd619d9eefb03e515154311d44bb7bff9cfa2fcfa3bf93d060877e86191104c716125515a82ae32b6a17c3d6c7295606bcf52825df4e18a7a171bae3d080f921603201f46ac85cca75435ac1f5d4b7bc119478b7a97516579878dec5362198bd212967e23e72f055fa40be85d6a5e549223de62c081cd307cd05b4d654dc28c4e45e50882a10e8f48275b7cb72842ad477593c2c052b98e0c97e02bb319af35d2e3b838d398a8cd1944c0d3bd25acc8599514061a78aad0a134d911fd5413fb18927ca9b70c82411b82c877cfd0d14850f8bbb70f6225aac33d829a849bb55e87c8aeb57d9b3f8df767def37fb874afc61717404fad9e7efb28991fe94ee7d2990cdaf6b7b31506157a80b13d5aeb612186a35c3bc989bfa17c793cabec4ff9c4b56de7f5e2b73bfe213a4b9926ca0edbbddc4554348c0f6eaf9f5b4d024226a388e403a8b8aaa22a02781d03aa84fb8d234ea6397cf9c0ab4898063d1f52fbbdcdd9c4509343db2174c6c01d64a0c99e6d66f95c117775ffbdefc80c9216eb6c1acbc47e0137496d1eb2bf46c391fb87e2b1b92c8069b1c4a6031b2c18f27846e6166e9190e96f7566a8750075fd03e229bb603b4864f6a2677e7a2bf27a24c6b6663d3bb0cedfc2c3348beaca367535a8e078f8e7eda71198be5f8ced1263741217baccef7bfceddac36e565ed2b1eaf6cd7df5dd58558aa04f45bf375fd6d77fc751d5bb65594f13f186e392565c8a89dee9a29759848624adc7338fb6d15955d1bcd6831f777ee53b80031345d78ce891469fa455b68c7c565e9a44bbf5823ae33650050ed9f5e4fdc4453005b57a79911f4d92d15e163de56653f4e3b5daa56868c28fb7e33c89d4de1a972a78dbd33ace2cd1510dabefb798e12ea284db2bddcebea1af35bb5a4d3803b72884198c1aedcdfe5927c5e86940d2408d04fc14bfd294d959441accc35c90fe9489cbea9e0891685e4eecde570897e91798c6168b0ef5954a90a357d3d632fbd2dc5b95507927ade4e599382d5ab266b6c2b197cbee5173f7ae1d81fb1f7a3d880e4112bee9e13d617e536c38b1b2680f15571eff0b28e3f4fec1e1a5a2c712088234c955c56ea5689b94012598086b64f32f2290cffc73a6f538ed700aba37f8fa8f4f918610647eafe39d608834cf7d0e92393da73c819233fb1bc6b1c41dcdb9b90c1eb696f8f83dba77660cb822abaa38d9324b79bc5b3d80d326225511eb525a9549d132e189c79145c29012f375687b692b072482567452ca2af71dd2f656ec61af6094ac87d0007ef2bcc5c05ecbe28f9474dd52086af967218a6ecf6339d6f7bbef3cb896d9709972ed5107ec46f6b2eb1f04ca05a3dd10c666d875b67da2ddfbc78ba42b666a1bcf9c3538925c3059bb602b48c291867d6616db546484f86aeb9411edb9e9657c6553f2d0e59e280a61e2c16cb3c67e03ad04f7e9f1f2104507df6ff3b6e4fb31e0af1a7f6a6f364b9b0157947db53c29e7cbaea4e347b2f2ed03e654c6db83e2ae080c827551e373b3565e5e74e04976c2e57023ccf838eae7ac159f5e5fa7f539f95998aad8d839476c66393dca1e60baccecd4f3861a9394228e21e1c3cfe010ed72007f865cf2b38def933ce5238e3bec65370b63e0838f11b09af5ba7449df6ef67877cd1a46a4655b2b06c41145c2ee79d88934234114f41d591ea63d88d1f8476f083f789b867d493f0fe9302dec3dab27d1594b42e24434eaa1197423b513f532c53ff067d6203a6dccc738ad144479b39855d10f538c87ce3186c0b0d54c808213412a3220089f25c1e07359d7ad55ea22bc4d1d0537968254609ba6defc69db97058e90d090f7f593b778aaa2d2d05a24ef27e263cfe1956e596d7c2ece118da4ef85fa7453c56b96cec126124f79720370e9f5ebcfe975f3b3deb7411c2836e151a8e587bb97e66f98e532829da1d032f41bce29b22eb2991247deb2711e55c0ee8fc89752dff462a84d768027e9e357cb35e19a1111c9b2e73158ef575478c818733a0f941e24346f7268a788c229d668b0c6b9573329c5c533bc74597b0e6b522bb07d09d2e44970ec6f453acb22337f24f4041f2361661ec78fedd1f2852f4df1a24c2e8eb3063f241105b8554af350fae006b342ff08665539986b4b8b884e6da6196ca21f1f3ca1f4558c01229aa0bb113dce1251213e6550108fe6600ae60822516edd96e8fbfe6fa1104f479dc0de0e25ca220645099049e2ba73c12ba759bd708f61b8f7ff920994ffa0188143ec4d3b31f29cd84a4ebd8b75442d6f04cace13590ed3c0a9f5489da4a413c99c5e44fa8ff4738f92f8f1873af7f35980e3daa6561595d79d90b327a9bbc7eb5cbc8bf04ea838af5637ac42860fb15f9e2d79a5f1166da14d1ccfcf7957b37ae78f9898f8ae4acc16733469e0fbe605cf10119ed491c9b23ac1f4f18e1854dcdf3fe1711847286a56442049d63d1fc0963438fd5b18489ee958c93a627e2aa5a655fcf52833500e8823068ed043bd3485f2564e0f0a58abdb29c306a0e9b349df5a51b192d3603d4e883d8559b04cb610d3000d6af8887d72bd68e293fb7d5ddcc5762302a7c75afbfc6be9fd035ce9b96ebcc7f9533698529c315ae623e746411fd0b5d230f40c22e7e81914ad4c34022de2ea34bb0160fd4e92ec01e39f878ed208069c0f84a555e3589f4b2ec9196d8928f09780778bd53d23f261baf4b628847281ca83e7db077723e8faf8222e6e18e587ac53e76d6cf06b2a902a8546a09fa6b3d84e9bf1aa0be082ceaa05d77655aefbed95ff7a46f0bed117b2cfbd799c1b196fb02fadd01a04f910a4d2a938b7fbe9d5e71e56d5c6b5a262d4428f35f2562edb68f153fc3a07b87b89271807b842bdff97d277abdae5fafc1f7c30484ade60b5c9bf8890e46f4d69c761a875ef1a04b548aa513bb5a6676de4eb4532fe2"))
                    (acl2::hex-string-to-bytes! "b77fd3d5f25b7a3c0e42c707a9bdf225a8e842582308a73526ef4768f497f47776c4ea0ac5c6d1933946b14907e58dee"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "1c471b3e2b5d06a6382569b12bd43c6ba3c1c7e47ff0aafd51d896fab41a58e00b14250aa556a842161df9f649327f64"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4d241b96609427aff41b2d90417050e2201b9ff5d7cb32852ba2f51178f7d4f6091c89a004c1a11a90f65d24324e1772"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "1405434caf2b81a133c800119e1016be5f57ea1baf92d681cccef9d27e1a954a2df60748a19243ade40d0b99554dbd81"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "399a7a212c1e8e51e644df1339d2e02006c1dd8e51a0c854217532c69692bec120dff30d21869fe26a4f4a0d86ebae3f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "e881e908d46879ce673e7e64f8d90cdf5497953a9ee75df95a0b2f0dc41c4ff8664eaf31e97565ca2cc85d5093e54bf3a216e4e6cad34e098288d5c69a98df38cca12293950214c0439aa3474ed32772ebdf95b0df42f537919116ed647980894302aaf57e156df7d43db911d1aa87a65a7b0e5d863d3acbafae5a4cc9c71931f3faaf214cb958b908370576c37252d51471b9b866ca050c036809a9728b9179031099dda8f3bd4c0612f13ecc3472484449fd8828f43e43b6468bcd06cdd2a99ccdd0f23acd84f9f755e43095f7ce47108c419473c0cdb8efd148802e93c309e5b8bc28a94e960803894615c6c34e19a0c1efa5f0389ab3d78fa98992e74ce10759c6469d7ed9c151357eb507a0da081d706e8346638e7c960a14b2d9d0557829385416c066366c0ae1f6d8fe975a5aaf2226f61e498f5a53caf2c1ac57c61774ded05eb70d95505516154656afb6587d5d7fc620f386973755ac7fb282870610d2a0e72d7cdfb65b5a43c6b5f1d64ab0ace1baf3685c796935a8b4945b818d249c2e79c8b9fae64438e822c463a3786f94b49c40b0f74911f25028f9ea9af4d8c5e0dc30ced02837356c202a0f0329e39f6032aed4414e59120ba28a36bb8255d420d70ec248c45cdb38898ffddfa734544d45e6db21d2671f40ecef47cd80c35155330a9e4901dcb1b2249d342038893be39c8fcd7e1b63a159cccf25b6364ac4e6b8cb69bd29df3f43bbfabcac250add8b0152b34272761ebd0bd2d960ec10b63d2efa2b8b828c0e53144e9d6db6e1c1ab3ad794541fc820d471a16ab1ea40afc15d3eed9b550ca54bf942df4a323370ad10c5789c6a802e06c5e4e093770035da0d40b6d5e9878c91307d8b2ae32b70f74bc567029bc6d1bbd14243eddd78d2f005a88e2188244c8f81a8a9f70fee4bf3b37e6a6ef9c48f5b5b649ca662d20d6b0e1d7d67780b6cacb66b3f98ffa788070f9988c8cd9ff34e2cf2f2a84f336eb4f0478c8f0265744f956d57f5fb7cb09d29bdcaf85f8282fc0c934edaecc57e3173cfc2816e5aede2e857f8f64a954a1cc79336e6f793c6cec2efb72480b0c44b5e5d282504115ecc75d02b83361bdeecb2c2d77bddfa462a18af356e551cc96a7b98c4630cb34f750581f7840536f575b2f3b29d58e08a79d47bd0a0b13013f79029a024cd6748455f1264e1000d161e8fced135b26d1b24bffa91a3dd142c5866b7c44e9056d2a44d2a7a826fc9c95110fa3b2c31c88091a1d57f5d21f19cf7c5eb36cb9da774fa9571424f9a919676fe20f8681e1a412a42b4a3b7c4efe93b756d00813c83b7af0091c331c88e06f321e711d3c18ae756693d039910359d8b3de42a5ab512222b562f050e43e8a03141f88d546141f97ce01ba9ec4bec3f51e6f9ba3d4f7b3dd65ff611d051c6ec9826fa74d24766ff066df49d96be3b58f43953ba1153e79019e28bee5d3b5073919796ce37bc22815bd971e581b14b6fdcfb9b31c0097da45aa999a49f49b0bc825e74a46a069d08e3eb398b936b430c71f7782831b96600027ac63c7cc20b50d4413bfd536dfe1b066df597b6364cb81af493a64cabc92c7a1ebabddef08cb5f00994e2da3b6e432c254c0fd1588ef942b993b3f927cfb1295d1e2fac1958753c7e265ae7b61407bddde7fdcc1a305a4e0f21572074a01676fe9a9bce869a1822679a6d622819ef215b6d688fc45b60a7bee59ba9db63e3cbcbd5082b7f7824098d972333d7a483c0688759019e3695b58e86867ca441a18f8a787ead4b1a2359b9c685420cde4c03cbddd70cf0ea80e047ad21821761d729042a053b34a99c79b1ce9a770eeed1504e16ea97019ea991b00173a17bec0d030194a6bd804d18d33d596bbb27fd9a10b7da9a8b1a7b992e922b84c21a09cb84495e681c782d1ece270cfef49d3330accfd3f0ce83a9d721b5d25cce3ae19a14a32f543e647236d236915d0cb9b71a8a5ea7f349844febd0e19f04ab249b011bd8af700c8c43e847319090e9c3b2e0142cfeace465955ea5de39834320608901b187c66e37c9bd96e1b09d75034b0bb1008121886da4b706734ae17c7558b82ff35f5e4daf2c3c250e9a8e96c0ab22c8a3d75c1be5a47e5d6a7726406daa190b69de3e8f3a93ee5d7ff66b3644f9adc3dea9658615bd3c11896a5b764e7f31fedaad701e212659a6fe6b3a3cd55e3509779f42b24515778a99cb880661fe8a710d74fce0ac0f076951a10590bc0241023660fd86bdd7ba306141b3d7ee50abcacbdae8bf2273800c4141d31a410c5916e5959ee9c4298a6bab75e6d33d9c1c73dc873ea2cd7d4af996f3adc2aba505ad4ae52bc5a43371a33d0f28e1950b66d208240670f352ef96185e9a7044f4ce2f2ff9ae01a31ef640e0b682e940c5105117594613dd1df74d8f2ba20c52223b045a782e850a12a2aa5c12fad484f1a256d0cd0872d304e885c201cd7e1e56d594930bb4392136fb4979cc9b88aab7a44bfc2953751c2f4ca57a8924aef0b92248a45a47052fb38b86ff8c6aeef9409f976314300402c7aa177f6bae841073ac86540cefd17eb1dac7b5f971eebbb41ba2cd37a507cbf48b42994e73e9c5abfbb76acab48554576e7564bcb771363b9ed40239bd66c0dd768e90da7cc358899544af3b9f09facdd1f9d47d445db4fbb295c9c47d0929abf3632bb8d0fc11360493eb08b745b83c8bb9cde3219c7b1fde4dec7defd94e858a293390777a38dad00ee5d80521d5cc221f17965514d3cdab64310e1abc7694c44871097a27159c7f1c67548f869ba42092793fe8af062d9f42611f069beb002828aacce0e9f0a59b2437245f5a50ed0d79bd86e3535b69444f4457bc5ff3f03f32a981b1aa101dac901eef7d873dce6f3ea9124dea19a622f4a15f43eed78c8fcc35cff651be2b38c7e45deff3aa70d048068c2e5b890fbf9741dea332dffea25f155eb6c642f7c9315117bf88defab7ff9b9b8e8ef3c9a5a2ed7de512526b005412d7f9f1e4aae6362dc5f52c2395adcb32e8766b9fff5c7b6be410757afe2f11f4e2c7baec6c183e3f0f97c9bf693914911aeddad4d76a8c83ba2c459ddcb90083262909a164e5b1509baf46fb9de34281dafcc3db79593bffa8426904302688864c26c64cbbd8f72814448b1f5a33bdcad0aa0efc0eb1f2a65558c9f720fc10c7fbc7d9359876788f205f8073c7b2ee34a3ffa37fbef8183f6a794dbdd4c5020cdc132aa424499d4d67070d3f0de73b87fc036aacb0b814938eaadda07508c51e9712a1e187d8cc2992441aca535a0d8a82bf4ccb0d78506a6af561e62b4ec1c5d3cc2733e8ab6eed491e8f5dc294cd06de3a2f23fcc1338ba561c1d81d1e5ec8384aa2e46f4bec5f339e5440821ce1c4b77cfdfda61a25c0e8f6e5be740004f16f93fc3a463fac48f9cb51bbba7ab389b68e9ca1330b8ecfdbbdb5e0577c1b0620af1b7e920f186f43ae6f169237a9208377f657ca1cd78125bd1a85ef1b0bd0f3a7e9f77e44979eb84d949833da2d50c53ff16793d6669e6ffffaf764643c178e824cef2f65909b00b2a98ac5eba6e024ef7718b7c0a4ee60893f7cdde017a39c482eacd0fa6d2fc4e597346ff37c994a8e6fb45e63f938a165cf5d8215023bc3017677878b9522acec44de5a791643dc24c081af45ce46d99b94e27c8d66a9abb60c987ca1703c5fd19ad0ae05dea685ba54c8d3f089deb1317452f3d0acb3fd6fd857d9c36645869c8051921f3b89eb5612d0f2ec94c9d13159f70e643a226132c1fcc37adf04c853ddab2e5095b5b603e2e332bf55cfe0a22a17f6d37abd7e09b2ce644d4342ce009b24ab1f55f38fc6ac47bef336b7878dcffbd3a5d24efdfb9016e4c4fac73faaa1053115e1091ea95845bc109d4b6df92febcc60ac9b8510441d6b676c5bc45ad796e7b8a2bf0ab3cf25559cf4513bee342e098e6f6ebf51b62820bf02c60517251ef69300dffbecf72fb021a2929875c50eb3b6838d53c5ce822b5322cfa9e973e7862b642f9b5db9abf6260fd62d94cad1f26150c0e8757572b5ab354059d8a937be0a34798f0dfee04e34a3e5a1552cc0ed163ed12667913fe650a4dfbcedaf57c6b7da838204ac17e1d2c6869dc0d123cd5db586b7a50b32e5c9d6757b2f6d8fc7babc161a4bac98d688e083c9dca242b9be1da94037cc93db9e32c93da7cc26e99c44f9ffe9e7ab56de59a2ccb18764dac2de006c23bbaaef5822a5ab5bb26653e4d8f6dae64498895d99ac4f4769d358f06054e073bfbe4a7e0ccf6c5508952e1ae2d3534baaa6968c0868f9a72ed55ad10b6c472fc21813985a10299db7fca1c6c959111f031ea6f30fa073d8c0b308eb4812475dab01732f73308a39a6e3d571f550bd6a3648e01b141a64e205b58bc97d0c880762bf92cabff73633bc91f1a2ce80fe352b6c6d8b819f75cee7220bf01423b334ce9a97f1a62dd693e3501ee2bcb07d87cad274cbbcf74ea34d44afea731f70edd9dcaa2c9099115f026684b827aae268fbc21e642fea96691e61a03d04c866396acbf504e50490a1001c4aac7734d4e0a68393dd4040f52844c206742de63e206ab19d000d36fb60c7ba99461417fbf8b35efabe59c015e9a6af08145840a0aabfd4cda971f7f9a53e88980f661b00b70a9b7b90e1349bb32c6fcea7ec9f32e0527dab85770f32774dce708ed6078b26938240da4062328fbe4c81b551cb9fc90bf35f1e88454fdedcd3484bcaa5434b375a6e24dd128084448c49c4ee8c8d6dd28012fc0d62f8e2d3e049cfab"))
                    (acl2::hex-string-to-bytes! "4f20b8ac2926e6feadb98e36c9eaa74b368ffdb545a83323dc0cb862378175e2adf6df26ae3bb1c23cafab91b6f9c198"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "3dd340c35dd5a88967b66a5602dccb8ec44478d76662a1e328ad5a93ba16639ef7da57b175c9bf6fa2b11e006dfa6ac1"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "478f747486422a451f200e63775b2fa32b2af6f88524bdf14a3b6bc8d3ce5c610efac145f76391e7fd55c44c09b883f9"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "4a09dddc5a9f00ff888b1b52e99b952bc9fbb6444c4d696a55fb4179a7ccbb874bb6c36031f479331a58348ab4e1c77f0790042d11787f66ad1595a4badeac41781dbbea35ff0e512ae74322a008993957336a1227edf580e8c561a290e5ff139be6d062b0e947c281e34f02f33890bf7f8ceed8d7d55322cdb5af5d71083d677ab907b0e8d9aceee7f26112f598367a47429b68695822b3268d86bca76feda4df3c6abc26f8c63d23bc803af05112d279fe76aa259de99bb9f4efa779439732901a006eb16d1c18ffeab4622a8aac30db67898653cd30414f93f0062eacc37955f9d1afd36136e541240572ec6c8dfa3726dbe53a0f800faf1294c10fd1fae1f882e665d2fc9d0db2117c527e1f3c9a53e274ad238c5f6ca318e0c164dbc3d23f5c467af765d8f91a5a20ea28ca4176f7445dc8566c58c0f14ddeec75f1a7b4f42ff9d48ac56f1041a7fa22329625f8edf39d1c1bd6f074625b952b893c2defdda5085dd343ebb9ee9f10e3e3aa6f86fc983c93baf2b1eebee70d67e9e4ddebc3647afb1908d35eebea6c13852fa78aebdb43147ae92297cd9c4f86de103f106d145798a63caf66fe31169c5e4c566b8363b986a5e3cbd7561f6272aee8dcc7a55ea81bc0050839f01c1464b72a03918cd9a6d381073517231b4e11a1953b7631ea746cb08e643f88527055d7f03664cee412e2577ee7bdfa361c4d013fe487da5fc373befd4ee0907c325608c7774c102d069b6e706933a418d30c140d606068659d18317b4b84ce41604e90d74fb11c5c89e6100444dfe96a7b382428366ba3a60b43532fe5930c4a0427b84745fbb818a4ab8362ff2b90d41db877b8ee0e53c714dc6a42a375103e25c8c840e292ad095ed210b3227e63fdbf7455e411b33f73c1cbbfdb0d92a2c6cc1908e9acdec2d8bf43ff1f363f288e6c5408d426c1305019f89a23ac528c8a0002bf6b2d48c12dcfe798cdc2d8a82e2940d7b1b5a0ee06244b4242071e9531c96b37eded3670af97a423e64eb0ab86002009a346b7413eeaad015fa75c84642f3cb698731199071e63e791622ff93ef8dce7d16270f1a074041da2127622ca3b5d8593474202dd6c0d2df787bd59fd195cb9649032c5d3c93adf34d2d2370cf4b6ec07f12f74d88a883d3f44c2946b6c89c78359668680f1358f3d6fda8bd16330321cf942b6a07fbe5f9222a67231f84958574472fc4ef5df75e46ff4e5c28eb1a1f9e8e292de5395db18a8d1bbf7c685b508d9bfefe6dc1ed9525c8a0e867f227fc74676ce3a0ff7f8f85eac3764dc2627bda0b527efaedc02afbd03cfc63e6c15ad9968335bbacbff5bcc9dae214412f49e22bc6b12730039221bc7ee2ad175abd46b1f89ea5cf938687b0de7d6063364d4e63d85f73472c6d3fd3e3359422789644303988efe68eb48f265ab2a286fc187749e3b45f91af1ebebd068fc9969f684aa7c27f0a6026022b90bc2c5b4b051de554956a4c8e002926de553badc223c5c27f200b75faedf1f5969f450b1dd5e0210a49ebeabac40ef0ce5be3c153ca11542cde98c2e54af6759e5097328de2e1b2d19da38a966f42b4f386f2a4b9e6ed78375ab171ddd192141e9c8d5209b0950350fd4bdcaf6666dbf462e2b8730eac973530eff332f65769adcfa1fa1fb39c9afee52f31744dedee277c5e7d0da5f1499cdd9ed7833db8dea2667f8b4e905af231d1d829738a4a64334fa072e8413df7d4cd688f5d1c28fbc2fd7095f4ab42a1906fd1353bbffec67e39733a9164f53a839cb79b1f264a3955e12faac078f611a10b94e51b339d8d8ee62124a785b98a1fa8167b47c07f2ff1ddf72a308de7dada31bda9b0a98941dbb244d18937fd5f0f9c0fb0759f4a381d408194b35e9a55b27e96039a66545beb4a48b1b4d03331e9c4fddb16f73dfa4e95398ac76f702ab04ec873171e716433fac755c8f2ed9b24df8b4baa8cbceeec58997ceb451bc25a3ecf88d51682d35c602b3e2424aef347381282bea4697e9d072bdbad540fa516ab15a55302eb21cb06fecb64c9d9f2d75b3e30b1f2a631d72d5b377c30ae4b8c202568ff582ce7e65bbc8de97700f24bd568a622ec05911147b210d64f7a3e114a8944588aca8e54daef164090a67a1b232c839ff50a24d1c2dad85a2515f54c66cbb7ed1f46ad8d0100c4f8cb5d02cbc0e42977a226559f4b102e6645e3f9a42147343b7a239e056a8c4a807b7724e43b4a384877dcbbe02cfc63891686c3aee68669b25fd4084473fbe766159748eafea91080eabfc03b051a28313b50ebc34323a8e004ea44c0b9da5c184e3fe756979b42cf7f21fb065fd3dd0d18d4ca97b38972431dd90c00a8a1f92579d3a8ad729b6f72a9b5c4d889d4026b33a4e882a432666a0f22a69acced4f0c2ce4e5562c13c728feddc6383f7528ea1033de1ab1b8e5b35d59ace5a306ee3adc0625e720bf1a30f32c779bdd41c0f832f413d6588780bd9cd09777a7a927300b99f2e70cd7a9490b045189efb45b141078122c0528c7f72249fed7672eaa9a3022c4cc2d0824dfbb17cc78e36767cc4d3261b244071bee92083b7b4e11e8c7fe33240d2dcafd4397b970ef54b5779f2db57adf91cd0d0270ade5d9e8ed6247aa3d99307b9a9333156bc753b038fafb4663d62a7ac50be132ad7689a43b97a7d8c48ccc82e1972ca8c1485c826c727f3bdffc196b99a8f94c7975f12c66f9571e4a8737f04a797ee64c6c5057ca6e79529c7522f0a554b942a45ca6de0a17e3cf420109303faac8425fabe58ccd5f983f0ba9ed8692afbfa9c0a321359f73bc804d92c6cf3bbb87318daf62ae0fb64a65a022cfa9cc1a6ae630e1d6cf918d6fb5cfb4e729c91c4a5858bf7e47ec131e5b1c784fe6cc15497c77816e088ad56bf43adff0dc625d701b43849ec6fcf67273b302f4aa1d30d9f1270a2d7cd5dc4fb0317c6a5381ff96fec0772ac7bcc0e564266b448039edaad8a204bb991d12797e2622ba2bb2632d77620fc8a69612f590c919b5f59cbc2aa79ae98ee9cb0877a29b6b66c17e28b219aac6ff00335d118e1d76df66d01ce7c254211deb6a3ed8e1f3af06b14fc0b36a0d351450ff1fc7b0b63b17705e9050cd7053ff1a7525fba0e176007791cbbfcd0bc2cf1563cb86ad15bef49ea2a6f42d61ea9903664e8e63c748fa910e9a15c7290875ca0bf0ecd7ae8b0d4e46d14066cb6d8212d828dfaf27a03b79f0cc73ecd76ebeeb521058c4f317a80bb5afd41dff5520e525358df7851469bef2c358b2a97df0f5c0ba68f985fd8b5369831d97802a1bd6e80507b1620e0bbc8f2f229d11beb275f25f8be9d994fdb1ed0b8b87f064c9848b07db7140f5781f20606020a29979e84160302a508695a3ba99bc43c28dc05aa8ce87cb9483554340f79fd1de96cdc9487cf41f9d3ad9a55cce1e7f691d86e5abb98b04ff0019b57b653015184f149561f6565e224a30434e09208f768c83a21e6953c1c3244c024a29df8f022af4d9487cc54a620d4fe9f8eb70f85d635d1c7b5e8096934852a607b5fc940ea2c5114660cf88df1c7f1607e56e3e903a69b14cf98fe7d1e4c41d6eb9f5c8a05f2bb9f652e37e1bf04463d577a6e6b16a300d0ddad02a219226d4d986799159b89f5fd2c38fbe8b11463ffc228975238b468c28630ff1e4d8d168c2abe682bd37765f662352becf07eb26e666a46e890623bb37328dc3ae117a45c4aadb112d5467552af2aaf33a3ca9951ec4763734734e16b4c9d813efd487f85a03a5c13ca7d8cd8bd216f05f3818271fbc8d73c84b3c2b2b898d672cba70fcfd3705176daa51151312152ac082ad4973cf3888b706546505f1fc21d70bcbd028c5478832c456e86b633426f094df78956d25fa833d4d60cea3d3089170fefe572f0b982e9e50a3cf25e27951f735310f26d44daf488d8af01f705fce6290e412e4bfdba380e4d3e8daae767712350f3b8ca206adb08b6014b572abf059183de676d2a7442469f53301f6b2d28e556fc4ff3d9a011dcd6ca7164871a9dcfb584c65f2af82e18684283c69b2f6b41cdf160f059c4be8ca2d21a260454d66c0bfbf228e38e250910d42da65fb463c01d9b92c7861e09a080e1efa68ab7d7efef0a94d03f7e4fa3ffd1d7c7a00637e9aec28e824af61ba1dabe98ebff89c9e0c87b5460250fd36d9e012832446b1044e82efc298c931c171732bcfd4d13d3c8fd14d9df7c8adaef532bc9ee352c165a34a682c2343e33889f0503e55a9d1e2df89ecc552e378bb287196103599070b2f8f3cb57293f887edaff17aa106fd29a20e0d4ca225e0f3da391997a9ae147d59ddd7def55de448030e79d5455d01694dda49a4740478d180d5c195adf65c2498c2c41939f638f17098740ec130a12a597b0c6d3703a2ecc2f4288ab0f660a982ca8be4aef1590c9cda5a6793163ef54b6cdff779a7e1ae0700a8e76142a4649e4199d1c2d3d7021e40a77039724d529a2c3ed3aa65c78dc377ed4c2d5fa9fe51e468511412356a4b67e5c932d94ebadc45a39247c1c7049ee6402973a1d3df341f81f32140f6bdcabc9fc2fdbfd5a67825a8b3d05062caefc85c0fd2bdd47fbb5f713868c49d7502cb201e289a549e0458a8a8e2761d193c9a9b4fd9080e686e10b08159218c1889f0210b11fd5e19e76b3df1ecb778a80bd97899e514f411b2ffca6e80fc6510d6e81d4a685d54a2b7b9d32f5da8e7dfe4019e5e95792814206919606e999b166fff7f2c8a3edab0fbe102a1c3aefd294baf571e48891cd83ce9380770ada061bbc2c151f61652cd0a5d46f9ad5e2b2cb93f4d499cd934e67f0cb0b4458d07bb50ce8e3b4a884f7f89ab0aa7791ae452a33deb6100fe22b75d2fee074bdc535bb56f2048e5e94310f046e83bc589db239027bc15034c48d36ddf7a70f20ff5fb5041321b56ddb059b4bd4a17011f5228069939b6125829c30797d071e68c71f2b7d13e232bba29926d6abad9ba4ffbc88ceedec4bcb637fed0f57a8a87f659da629baee0c323015370fba4e45f803a9d89a525d160db15ff5054e33b54c2103ae70cfee4bb9cfa2575dcb2d77f74910f8368a95cbd2b263cec52a504eb74376c9ba1c89f837c5680c28a4f13988181a71085d3d7204b0b402b5b3a679a81220816f67681a965b8ee1d90e439193baeeca7a93fd118af179193fb0de4007d7295df77932b9bd"))
                    (acl2::hex-string-to-bytes! "6c3a8559da6994c513896859d491b019b3ea9dd74b8c39dc983a8a0c0a277c6b9a68b00a82772f48c873934690fabb3b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "3897c9713083c6795f60326e10dd4ca80b2fca125ebfc0815d1fc444e84350fdecee68af2c6a7c17427c8f40e09799525c502aaa6c1251302e13bce56b8620bc1c13dcab2d84d90182c1b4fa38fe1f912d338495c2ccecffdfa22083382b0d65cfed6d8eba39a4df2715e8c59d7aa50e85c020fbe8565a967ab63745d898dfa00e6665805cf1a7583a6275c99973f5706cdf45036716639c06364a54d2375e14e0650aa89a14a815c18fa12db351d560ea783dd5c676ceba8c0aa1d8ae6f526a4dd3a62ffc0a70bec784b51a747c0e8608734e9a8ddd6597131a84f8f5a782495d967f3c0a3bc7fa5129f8b7625d6889cdb8c17a45b3eedca34e3bdc2e49b75a1997ddf84d1074716966b42d075aaa12422fde9c755f3230448e0dade1bcb94588be0a8b4cef41cc16232fa3ebdc9ec86dc1ec29a54bf93cffc24493966850b1b2f44d1c8f41f4b36148b342944c0a54585435485dde0ba7dbd226b5d6b54446a01fedd39f6f56cd50bd86318c10b75d0bd1ff2226802f1ae1a6fc3c280b5cad592833d903df2fc3ee6af0995b926d8c8c2193df87992f3d18260d4f15d2618ee5f8675643fb473c40373ba28738f389e52a39a60991d1b7883ba31724721c904d76d171e82a65e1500dad739a6c13fb1ed6389f32d1de3139cb04bcdd53525c9889b85379d3535a25d290351c95938a3d0cdaf38dbf1d5234bf7965c8ddce9ace1b66247e60d74ec7702a0f931a3cdf4cb465ca9fc458c380004a3a6e7957f1f813210b8038ba663fcdc42a8965d6a252b5224bf249552b2575bf64568db4091d58323006c3c34994d3a5ae85dd297b58f9599f0db4dd332c05200b0883ae8b45d4e1bc9869bc9fce6817c210eafc10656752ed786eb8839c9ab4560dc10d1f786e75d7be926b12f676608eb1f4190c81e7545ebce0aec27d1bc46eecb1996cbf0e38a801a69a4812e4c93bf0634615b461a81a60718798d76f987b2db9191b219c7058e80d0fe92d9a9af155a04cd307bd9748ec880aafb38078a4594357d3a701660efc2f939e4c801bd76911f3f081e36711e88f01ac1835ac11f7b6193d582362f3c375ed0f797e9416b31f02112bbc609cdded323a0f767beb8f7a28013cebab2fc745e7e3bb7c4cd5bb3243e00b2ecc12530a65ff9f2b2f5aa9f74cc65b418be9900847c21b628c3d0ab37c052c22f19ad04e64a3d03c0e85ce954c34be0e484aa427d6569f189c76270876289381195dde21f6cfaa92619585bcf2f46c382b798d085c445186c4a79a9bf9f04a318a8204ebe22b36136fb157d404ae6da97c9ed315a13739954cf9ecd3cd05581c8c32cb14f15e7ba88202cb454b09ca4783e3fcb56cd55e2b7b93a417b910c5286995b904eb11813ae0b069778fae8afc886e924f86c6c8319a9d4ad186b9993f67d2582398eb51afd2e2b9d29845b5afd2671d4db8b37f90f1416768fca7dd48d01230dabf95f2f1a0c044bf2d755448aaf72316c8448c2ec8a58244e383992bea7cc78f2ab66ce6927391a418e66dae58462e127e20f6f567f123cd03f4d3d0fe39a7ab9ff7c878f02328ea1f410bcb247bc5a9a4f0c5cc154ed26b24a3951f53705897e7000d8e01f792fa0b2e338655be7143005b019a1ce418860d37e600334a60baca78430f0e18563aa354efc73539b8485c0ab349c2da7c868759cf5be4ef497b0f0773e55faefc14b15c46760a1125207ea4e3c0505887d35ab15618af869b448056a4f73ce00deffbed417c1d713d332ed297e70d88bfd54ee6861fdc37dc7ce30dde2dedd610334bb0e6bfb67f8aea363cd9a64390aef42aa16458b32c968a01d4296fb9b98d47674d3f281d6393ab5d89b0e53898dc99b889faab7c8a50d8a2518ea5eab10c95176b549f9946e0df6a62258c36c9d02abe474f017e7c93cd8d2f14b7c6e5168425b22b0d7350faef266e3e83834a121105826b400228e6d18c14c48d967e327c57ccce58f4a86841595605b267442ab455e3575250b6d14c65a9a410a1a3e9e5775e456a6177b23fcac182022cdfa17d76121a1db04fdca6e195bf26317dc2284b13962905e461504358bc591842fdd32fb6bc8b6a1b9048c8801ab88c3445ab9c90cb7f2718fcdb8ac3bba17596c91d49774683c5417498e214bfaffbeffad73180ec530e3a0b4a69a987c089028ac58dffbfec0ae911fff7f6b5f1c6d66a14d489ff8c03731604a5b229a09dafca32c0f32fb0823dede6ced1046835121ccb4b530e144eeba6710987a827f8c510ccadfe9eb6726292b3304bb5fa90cb98625f8720d237c942f95cd00c3fe46be5fb1bc615e6258388d0e50cce911e286743d8cd3eae0636730465345dcfa7c9d027d1f39b28e2c655a3b5bd5a14285210d47ecc6f900259c0faed5ca4be8a0859b8982e3964f053839668065e6bf6711b7f192dbb60745bb143bcf0e9f6d5500816381ff68298539a0fc58ee47ed0ac11ab70972a06a463ce479a97325ed0c866a9796dda6a22ee207895fef335fe2b79bf4cb3c56bda9aa06f4ff989dec399fada23c0ed05d64b9f38a0fd9478c717db25752993755e27ae21a2bff55ef5ea4c0ffdc23839fb939eb7ae34ce2f7598bf44b3a09c0d773380c65d917fc6afe4e5b1d019c2efdbeda65d874ba9424ed5ae84cec1e91c6fe69fc746162dfe39af4c29646d9310bc823e92f2e92d999bf7304d8d3381888c67928a26f709efdb7209de8167b482e77aeebeb9b6631ca325cb07b99e6d3b17dfe179c4dc1b7fdd56a537efae2c2fa4fc3e49a25c52b4b088091a1aacb31b577515fbea6b6a978682a79e75eb03fd3b30b79b76800247084871ef78035d129652b5775e4ffe201f1e977addc642d6f33fe01cfccd7cbef48c8bec9b4b72071a9d4957e92edf00f054fe32ede6f0a115514135fee8c702bc7ce07a4b2dfbe12d7f5a0253b11621414a63306ef41546e98d634c427cebfaa841f7024f50ff8a15206fe0647fa9695a284edb2797baacb71ccdcb922ff18d000358ea705e86e91aa71059e441a3c28217479be278e5bdc7260718b4a9e0a50be98c9b76fc99e6863dc5d59a9e0913033fd7d8653fa16060d29a9d7093c029121c1a7e29d43e699f0917bc4f3fe54423c785fa7993f37dcc480dc090959a005dc48afa508e7f5aaf348d76cbcb3e3c2bd7da69e61ce5fd51b1e8ebda56751434076576b5d264a1fce5b06bd6eb1de563995d8e74837ec63835b485187f594cdf10325c1e46f427cf45060438478926123396cfeb70c6c1c9fbf8f610b74e90a5057913549537d26f7c19a7b8868ae781cd0d95128ac6d64b2c3ca58ba75dc8d00fc3031f65358b26b171bd20502810f9904b49b27a85ee08ef09fc50f78ddca6559694465ff73b3a0c94af364f6c8ac0a3d75193e70bef466de562294ee7b570007d2e932a84c8f4b60ce3c3285d3b1e2f39177d7a61c34ecd5b3fdc5471397537fd39cc6ee41c9cd761c38e86040c5d5cfa0138db8e506838f0fc32b23551ff23a01dbad5201d7a9619b6610f67e5c451a1798bb19a80dbd62efff3103be654ebed4225053fb5d51cbe23026b90af9a37b6e640a58e39f1ce9cb1e71589debdb96c11d5a649cbf791d7743a4387502835c1b01d5e7165f049c27ad56a292382b2a6f091614746261ced10d13f52b2b6744b3eeb854beb216e29c387b7239beb00224822de4010abe5875f8f06fbbb8b1cdbfa71c125b3cc6658f2c3a37d3040cb688251c43c10ce152dc93c418a55f154fa4b79bb06a53958d77152d641c9c83a0104e4bb432ed6d468b69f9705672e442f99c91af6a9af6d2fa6fcf9774dcc767438a36d5e520d49be8e81b176d98b15024d2b87fda778cde7fb88b425a694a74985587f9464b99db9b9aa36e49350effbcd64aa2487da9f87d6c0ae55bb68f30952c99cee4c23f7e19fe2cf587cc5f9a3823f2a72311e0c98a16a900e7490853b09bf6a72f29380bda9f124d97e325c81d483b0f08d8b3ba1c34f0159b95383671102912a3f5060d53d9edc6f80de3f1e2cf7e3a4846b542aa437aa3d72fb1616d25aba1e19a4beb1c5b2a0226b12cb00f0656c98fc0481bbacea32c287ce65ceb48f8d556ad5c2daa7f617fb251628b18a93f2d4dea0e8dd432f43f0e015991a1f68b1d7182c96d74d2fea0197327db960491e5e1385b117d1cf82d06617d97f7bafcfac0cf15f301d28c03da9aef29edc51984082fb05f2be939969ff1e4e87302a8edd2f6c486e81a842b544d1b1ea64720782124c0cf5724030c394701fb9a6a2977068e57618e07618734ee4df246273132aadbeeda856a1eee70e56759d45de91709ea314b6a15be97fb12fa7d53d1d65423a9bd193d07d7cdc531878bc76a825e2bbef48d02e1f5ddf53fa2f845d38a5d1a029df3f499b81aa4dfe4f9d9fc6fbd0a6113aefee8a1d9d1e8d9fb92a8c7be0c9b140332ac4a65ae25023c92a01ba414751149a4963d1c1f82f20fd0ce08d769c7a1bf7d9c093f3fc551bc67d210a3e86f6fb5aa7742151cb2adad66bfd869b01ce9914c057ef7d8256601e02ee95ce4b4a19b67083b3577ec4f4cfafac0c57e0f3e211455163015c9b19581f3000f97e9396360c5932daf946966d40b3c7b321321d76d2759685726c6ffd3eaef6c7071a9a62e4be44e324f8ea3a86da61b6add372ccce8c3114ac118a94c6311d41ce274cc35e7a9179e72115bdeaeb7f696a1e0a99a15edbb3f2d191a51f811d6ebc02d4e0e354c2a42a0a427ed2acccd2415b0d68b2b33d6c2cf6adb9b0669f57b7f25b41dc157f6cd5fc9706a7e7edfb1ad1a3be6b53fe4642c60557e3a84b02a840096faf856fdad03d3cdcd2628decdec9f2253306c74a1c8ac2071f0ec879783505f1c66c3bf503155a88b9e652ac738b25247293407ea2890b51cd61bfc40d17784f87928bc656ce1993c63ae442c35f624491c310ecbd0d09be5369d7e9512e89125cbf0553b2afa0846028ecb9b58dd10bf65ee09c535ce3d51aeeb9f22bc461c1acba54a6fdff18e3ee761216305c9c030a2c57944dbfc21e8ab80ce6821f27e4de4c1ec72107a839daedfd631e16e8b435f41ede947ece220b8e79ce408804045c3d0cbc040f790c8b5556a86601e0a5e080dc820fe12e68063aa8b14c4720a88d55c58f905eb3e76976c4ebf065a95e71963a86a5300b79e18240978ea4a258d95bf248ab3e7d4fdc17f036a48c6e6752ffe0f440b28cdb062fea89eee7b29ca8193aed3ff5ffce4c0281906abd5549e98375d3c7b0da18db094d8d54577afd6255468833cc62eb21fd20022a3f868ba0e9a22c21721cb8cad712aea83d1f026fd8bd206f0bce2fc3d1"))
                    (acl2::hex-string-to-bytes! "be2da5a9473cf79f41f123520e48c0b37d0b78af0e640f6f85892fac68489e29b9992eb38215cec107a29f95cca5223c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b236f326dcfab7c5ca01b0f0e172e77c53ff4e9c3f0686b0e60b91d7ffe134810dca596fa2ddc11827883c1613f74960"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "cb16838296482187bce81d180666b720ea763b9dd3bead81134f8deec96c98bab0e938651e6c11e6fbd76fa241cec784"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "00d3b9396a09d37e126ecceb86f5db9e8ed94065646f4d3d6bba15e8318ca9f6d07e363d60dd863ec28ac2378ccdb515"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4732bfd9941893d6597ebd157588b7438daf1df027172664f8452d019cf90bc36ec890c143016d38c2b8f763dda58f39"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "ab4cced009cd42ca0a92339d0513667901babdc49182fa1d414cc91a48827429ad8cb733b93acd82e0edcd30d0ec568d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "e1a1e25e96b2ef7504b76ae5adefda0114c7d4a7a6c6796745ba9c9d204864156b3429fb9b9aa1908038f7928d48267e"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "6f07c911878a86e57375528b8cd2adda91e11217b8c944b20fb1a4fdd6ae52bfeca5d4f929872196a6f17aa1af942034"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "ae3d50c6b6dffc7062dcfb7788c60029a467c40b75e37f0320f001c4588e83024fe4be1cace1ebd8b1da1744a920546f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b59b1626d9b8d6ff9a35bb5cd69d29702dca9471a3140e7c2106b295eea3193a850ca00d15469f6d355bb60f48a284e3"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "757a852bcd6479d42065467ece8db9489f081568b54fa7d505ace1365ce78bab67351c10aa65fe0b8ce42291ec9d2132"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "46f2afc4632e2121e7ab844474534f44c4ad9a13a0b59eb86918aea1e16bb13f8e753dffbef875bc7818244a84e4e33f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "9a7e7f53e213c84fee6961a65064a3c426c68ce6432266158c60f7a3f87975386301ed4c89d222eb55ef6efd7dd23ec9"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "be87bf6e103f66527b12a010e50eb157524c76f0fa7118ef8f0659f3a8734008f973bdadfcff5d84cfeef0914245f47a"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "7eb077392096a265d703e730e8b65d65c5eaa03f8fcd777bd933b4b0af8c5ce3d613085656498ba236a2d505877e18fda45a2916b74828007f9c63e451e978f85d2cba523346d6fa86b0b7422f6aa65a7434b61f8b015f345aa9695481de0be69a6155d2bf75cb944d95538fe86251a3c0df100112e0643ef1c08da746535fd5161ba08103bfb29916b87192bb4e5f010dc3ff6554ce7e47cc4b9886a930baec57c36404060ad2306683dbf6af16199be5484af35ab1f67774b05398b05d6553e7bf6906d1407482c75d5861cda2c79acfd6bb522b7314f57059838665c2a8334e79772d6a47e0b6c0aeaf7c1cdab12554d5182bcffa20ea48acf3f7d048cab412575621db095ef5dadfa9eea4a71e65faad2a254f30627d85136a94f8d515fdbdc3cd8b2c6c36679b560e5f6def58814cc6b9e9b2fb22d7aef5af05e552cf37d724a613635aaf338d7ca679809ca5a2cf898d1e0045a71f4bb98bf19fa84189e6e2a7608aba128238b938e1e4944ef00d5c6d6d42f53a7ee5c9c6002fbdb31c7465991ba873a77b23768e42a1c19a928da8227e7ff3231c038c28db9c7b6d17081443ebc66fb647d6196e42c009d3b1d7d675f8bf27aa202d6a792aeec9b6a32b596f79eb95870dd90ca18948563f74744c2b840a7917b3ebcb80942d8ed8e8639be36d1fcd62f6b5b5efc0418f2804eb740fc969d1b1fcc61706c262b4190d3b43686e6aff8e2af17e1f8fb39cd488c4b5e870e317c12445b6d35c9522c23b38b7c624d51354d015cefb9da7658c2988767d2081338f52f111af09ccc01a99e460c53867f2fc10a90c499324d1970fbe2197824a0d63af508e627111766b50c6a62d522982ce1336a2d21833ca3f61ee1dfe3d468d8563ec96b33d6b2cfb8766f2c51921ac110144f75ed5db38ca48a0d27afa7d9c3c954d926b15e3ca14674daa563659590f025973b572240181eb5da8d361f820d26df9e852a07f38fc180238a850d96f0e3cff7e9e96160eac73ac07d019be028d147a023087495efbc10d028310a7d857dabd80e2e9ba47930716573bcbd72bb6b25846cd7149255b8af772af1ecd7a32021be1491517947dd02200da1e4c82946e191ef1078673629c47eef321533fe6dda2099bbf35deb4f5586a76bb24ee717147f163f108325a1f85e019c23f0dbd92d961a0911d0b7e9f40fdd649675993e36926ecd6d748893a3aafe7a89bcefd49494f2033d60524f66322c4711daa43eb0827545ffc9f3816edc3267b6f69ef2c783ce24d68b2ed21483f06db4fc9c9b1ab5b63337e1daedb26849de5c1c6db60f091af86dcfc234977be652e059d6d5eda2ae0fb65ac35b06d5a168c9bcd091039b44f8639ca0c957905a9134510771a1efc955eb93313bd4bad57180466c062d90f8af68703f22ed57e9a3119092e0cfd3948058399912b47e52a3205794a7e5a08287ecf911b3b138fff318871dc5d0149c297cbc8f722ea6a5582a4eeed8d4409f0c5185bd0e5bca2a2ba9a2ec734b295cbbd97a78dc9baa4a0afebadf2b71d79ab746a495854571e4975582937ba97832b1ff11e9a4941fcd562e29ea1236fc02132cd43fdaf6c6bbd09c94a7617387ca964e01568ca07d8ba4290aa41fac1461aa84a31051f83f1c61ff67acb721de90e346d743d7393a119f8fdb71c639c725e7ee8fbb9cb25172b5fe59f7df3cf3022399e23626c6dc0b82b31ae788787aeb45bb08a61dcd947b670a0a2d5c4d5bbad187d0b1fde1d9b5f8435789e8e0a62e6e7a8c4457928e92dfb2f6c1b0ab7f0777c1212bbf12b9cd8aa10fbd69fadf4bacbda92accb996359a798e497f0fe39470b642180cc202aea47ca2167e2f9cdf9af4edde0f9b41e021de5f1909b31fde76d7b21b49884ac5c006cd19b8075556b3a0b8b272c0182f0051e7d5036d8a31d4b5ff8a07cdd8cf77ffcf3b8a1aaf4d7151985ca8c09f8ea06496f1de77b95b24d19908ca1b531836c856e076c1e1306e880b4104053e6bac870af40d6a33a6e1aafcfc02e9303c86e0c2cd00a5b18c15a48ff437167b526f49a24012dd7fa185b2c9f834c006a8d2e9b145ba3e96dce137a4da946cecef732d0b8a711a03d7ab795e71f2383ce9c54390eef5e4e78f5bc1e888ce80f97f42919527cac856b5ad7f48fdadaa497bb87ef02d56314a300fd3a97327eb27ae7827ddac6260eec8162507efd9aeb2db36fc4d856dbc255b57dd862e43140de96e40439cea7c329b701b29ecf8ae1f94bb5c078b8eee1f5dee5ffd1e929402bcbf8b755dc562d88c35c21d69ad30d614d01ebcb43c3f8ed41dcb8cc8b8d3ac84c669d17375bd3726f1aa50ea49cf7ce7ffa4eb741838fa4a70b0a3b1085b61fe1d5be6729df3d5081e4462e370e28a9ab49f025f8552f82d719fd06c13b3d9754ed232ef80af488c2610d1f995127589796d9feb1625eef9baf9915e6d456986c57d352ed037f84fa3a01c7128c7c450c881228b4d1a753b975e750c77b8a8af0bbd6bd50f366aaf1162b0fece7b0d6d1c7a28d49cb8f16fa12eea0ab5660d6a9c03555690132646e9e91df6d8b4a75b0fc9114b878a0a0262fb6ebdc8d60e25f0ff7047f566a193b7a3d20dfa0d2896fcfe8a448ff615a0573481ba0491217d95024b1dc1996301e5137b8884dd17ad7e7ae2743895aaa5e87ab1640b2ffdde12ac544c6733f13e0cb8591b1bbceafb0edeb84ef3d14f9fb12e7c603efae9d3d45e1a4868cec2949913192a1125192c288a7a3954416b0a2bd05113329a7bdb64a80bb648b25da204b43a2a279e28ad461eb6c7cc5e8351c300ce1168773a714e1aa70a798728e2fa1c1abfa180affb01fbec7eee913f23112041df63299c11fb168ab99dd5cad3c39b594f21850591c6b299c31d81e4da80cf6ad1951a1759f63c154807f6a9f10be2645d2a13d01e1ed33d38eb991b04bc5e979c80d142a5b9ebe54a5a925cd73d61da5e2da585dc74ce4544db4f582e0e6e5684a8e79471c0b66915081139d2c29a19e6aec29d695314407b6f3d12b3016f264e9fb5836aa52c1f4c92fbe780560ba17b1ff38d515435cd4119bad1303fe1955eb7c3a5ecd1ad76557dcaec7c42370f900390ec00d2b33a4a5d711a221456121993ba5274d0bb1cd43b069f03219cc2a9fc532004690d15dbbcd86405194d3b13b5d24f9f1ce5612d6d2795d17ebd593d6c92cfe881b3fe99ba593555c9375ed190d3c3c820bce1e556b66743b41941ced6e85219a74842e8d88c1490be951bd7a717758c96922d70ba2b69f65c10fae2bdbc4d04642a816d99e9784243f6f0aa8b15cb68b9159273fa0f3455f16502fd9397175522db488c60e9d6e6f3cd5bae9f673bc932b7bf5388ca4cdefe65f893faf147787dfaf6b9d015b6fb6898db7866845d3a8de91283f8f52a1deda42a2a566363eadfda1a9a79e5af0bfcd9fe7c72e59639152a1aa9cd18428f399036453855e328c329329290e215753b11134f37137d51730b6629779c2caf8ccaab5903515bbeb238bd36e78380e3added855458688f0220d53878b095efb31a3d7fa69ee241947b948fbd05287b3cf83bc1300c73fb1ac90814869d83ffee2cc6a9dbe74d2c4de0b3455ddc780821aa01d2d0b4e08026d456de0a6916626fa94c1b7839dee4dffbf2dd9727b01cb09d499f7c793885f0e8c87e4c3bd681ff3ad8bdc4bb1ed11273583fb30b591eb2761a107b8302743eae64dd943c4ac2658771770996945badf0ed54f9b2fa78f0e07d0d17f9c6b4d9ce34e12a93fa8e6b371f318df1c9eb7a1cba02b0441b912f81787e96030dad89ef4073556a6e4ed7f2ead39af543e73f9cc63222e844d13a01164b41cb5dc715b98e7c5fe62f687262db4576c98cb9c51b1acd95c7fea0d8095c5d6076416a47d629e28e31c446042df96f69972cc7d43fbcb95209d066d041549a3b2f3efee5eb168f2b73b54e55fb4d39bd84a97b34b0b6e0a1d5438beb5005eb191cdeebb8a35e697dd38bea2c75dd8cb8ba8d3cac171028c953de77f1df77384465a89988a30a5ace7acd815242b34ea4615f71051cd07b14ad2993aa95b36a023324711d9622496490037a900bd769c7a0a884076600c0c808ac45dd1bc0679caedbd93e22ac1d893cb53fac784fc36aa0efdc7a9ea7df2e8ff0f65418a6195afd543ab5efb2c082a66c376e302546d489bce1266bcebf8e677c0fd5356e67834917fd48a63b199af7afcaa0dc61d94f8298fbd3767c4a30bbd8f6291a1f3861b9ad88c0ef5010185707600823096321b309924867b9675bb18d15186ec1763a769d45e471f88d43edfb6170ae27ce9345990e4ee58dfe597122d369d95d0857a76f9a7ed53b7865e213359e25c201ed60d0d7ad9801f870241f97e605f9d60075acf154c221b9c45dd7455a2ee3e4772b35552d291c3be9f20dcb5597a0c87c08dbb130e1b339a887d0f76e74915f79cf50f19540e7b00c0272d67f3a57c615878442d8daf8de5b8885b4a09dd45a3a74a911af84683f207dd9202c9428b8111a300c1454f89124877ee5ba52b4bd0340c91f419ea9d16f8ac711234d1a86b9eb6577aef8bc3b4aac44f0823a3a701c5e37811435b24c3f52c6112e0e3518551faeeb6c6d4bc216215e3fc1fe6d5bcdf47ba1f0f2899ad9fd5d88137f82562de190d0185bae10d00b54383e4556675e2b1ee59d7d38a185b215de9b4708298857958e786ceb0bc095af45c3b8fb7a162aa7b7e27dc74b8a99747b8e0ed077b0a9448d55460c9d2de4f6cd16ff667d3e1a9e7e889b3c7e32db04c7b47ee02aeb002535769118ad6123b13800c19f591e228ba1587ea124c3a04d2bd53d42288284c1efa5f61822a27869ed49a6acf4fe6bf78a17cefd24860fd60db8f3fe5bbd5c9f21a81c13e6390ba456b9589738261c6aa090355966c35127d626491aa9e3b1483548ad11e06e05edda187c680cdd5e6a4b8a59bcd2c28dc8efe7e6a0fc3eaa42914d4e1bf8888ec9e12dbd25576d5981f3cce22e50caec42bec374973b1470179f7fa93716d1cdc2d3c1c0d8031c3436001f79b8c824961bad8298084eae2a4ec5a66670228feb5f199c3088f7331811da4bc6602219b2de23ebfcfa1c5401706d892dcb040b10281986b33e1bf92c9bea18703046ef691088c748758f4d1d04fef1a452a58d62a63c47f10157b8995b964e20301ceee23b43de24828ee3a592e0a6760ceb5c0c60eaa694d993b686da47b47d678d6046e6bef83627025cc1a37a704901bfc234df8c6d41ceb73d1c9e661330dd7f8bbc9a60dd296ed1019bbe3617c63a0a0503a98862a2e9ab9382315933e09adbb96dde9154a35fe41a586b7b6294beb26ea048484e5f315700e11b8adc36a78a30803ab9274c0123ef2332b21a5b95a145334432f820afa4ce61e25fd6d90b73381e36401cce5c76c7229c4107c5c8bdd2e623b02a8c697843291042e873ce7d1b6f5b8ffcce78c6ec7d3091f1785ef05cdf6f29aa4b69bda9622c9adca9f9921f9a300c8ef75ef9a45d760aa01c5f202d46af054106b7594633bfcf782592998ec83e209cee15f938eaf714dc5ad4378b446cabb765593663c55049858fa479ae200b33ec5ea5acf04b3a186cdac41cbf012d85d40f593a4740676fca8e341cc8cb21ae48ed17fc13a9c87c194337b711a47afa60adbe4f3ccb8dab11cb4c6923f008f7764372d0eb2b27aaab3d35f0c855db7222e6915c888e486420eef34809b846783e09be0fbefae108ea72c7af2e3ed955e363d94874b22303c09e32a1c04b65986ae703a3d53d7e009d87f4dc8984fb0c6bcb9efd485fa3eeb983105c91d0edbec0844a73781d84f98e27c55f2252cafd82c7e9624cbc528ed3469562c46f64540a899a67eec27d5b83f8402041cecd91f0ffc15fc8f2b74bd68fc999a15254c10e9dfc678153f08d45a26f082e208c307c1a2edc03285a8e8905b5ea709e446d9569775b77f866f99cce5afd30718f43f0c138c14687c4bdb551603d787ebdc06695132b19db66064f2408e890e95f30129be8f6d9377c42221bd27b021846224bc8d8c0af21781c284b81b297af8361ab514564045f0a83980a148a569467fb0aaacf0081f524a3edeccd418aec833c828a57f10f28f7eadfc62d54bc1ba8b36ce4ffd4d67df30367aef57fb4c7e82b61ef278641b9caddbef825c07cd592b2391e7124f8e16ca55e3a00535c6938950e27bc0ce33b0bc6ff4805b5dde9ac6039ff2bd92d4a18ef9a37f6a4c9f9fee252db1f693dd5430c685dd41d287f7e26333037d9bee07d4ae24edf5c44a0bf79f5b3568797b0f917c2e09adfa6c1916af1599d0ef3ca251fff27c462158605c4bf566b4c39f3b54fd4785fe39cd56c953921635f133198d4ba69ac9e84d2572a786d65a26ac7ff1c68ce2c1155e39810d43093b3683c5cd3d9bca0f8d9ad3e13e79f4d70745460e64489085118149b428a2f34f43814fb8224caa7585b3e41e7d5952a9e9dc15f153152fc8b2c8fff9222c64f09cce0fd4b6538c9a653aee80f0af704132e378511f7d10fa7da7d4b6a669cebda6cc9e89562b10eca0924bbf14e6ce4b96edd58425ead498d6a0608311c5a104f5ff746c7e4db7facc5c943f6699cd177a7dc62112e8fb2252451bd4e0da7c650ef919e4bb35e2335cfc407862230d3351f5b2be42e1e53584f87f22b012c16c562bf798897e3034b6e3e8f0a42d0f2db208339242703330c35cf302fc091079601ab5ee6581225d7ec5f64e892233752f07b163ae2e55a2a1c52fdc2117da2ab147ad5cdbd9293c28f4e0d8ad7d2a980d76c4a8c246dd19b4d38e44cbd98eeece35766a5900c55f2a0341256dc84504e1f7e5b84b8439f4510a86a358d8bf342b5689ad6972771516638fcb668e235a40ed8d389650ecc12adca19e646feb993d4d5f8c89b8cf1c62b83521142945f6beea33a02f27661e96bc778c5b39849c986c709e463e2dc38353c9ccf3341d68905061659282565b193c70f3b9b02aaca14665390b9d23b91ce8c63570391a3e89c2af1f5cfc0e0546df167b1c5e4e480c6d96035630415b83c9b2d137a2ecd6efa37523eaff2dbb9416e078dfae400d1e349b83c2a1f7295c790856a5fbd5c056954c29c8f605cc85766773802ad05a7b5dde10bd6cc5b10b994f0c47ce959942fa9c87738b56e69f600401ba5e5fbe982be650c2fb2d61158cef8899757cb03955fec37739716846864e99dab35ea7f9313cca6fc7aee80ee67242279b2ba3ae39eb64406af5e01f3bbfa25db44db5030059c4007cba845c186fc"))
                    (acl2::hex-string-to-bytes! "0310e51ec82c5fbb0d8a3a3e520a7a0f77504e0d4744f36b4d87e9cd9222056f45b591ff4f0b752cf642cf5859ce2682"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "f2df4b7af9ca9da652b5027a6a68ed91022eda553d72f61e2abd9ab46038be92141d3b3dac6c0e2f87db9350019d9f1a"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "2b73f1ea458d69bd87d55e2c055936222a83d927d4c4b1b059259633ab0b14d0ac62ddc4d04d6a6d2c878b0fa1b53fbf"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "50e223e4f32f90fb64ffbfa33f0ea65841d75dfad19d0eefbc67664b546e60f5564bde3351b3d49df2ac3613c15e43db"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "0a53523921e16516ac76648367b4a285bfce6b711b99e4775c00ec9d15abd092d5db3ddfa3560f3dcf47f44031e91155"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "9d7af269d49bab3b400d2b66254fcabe81cee8bd8d0d2b44c3393bebcd776402746773318ad6509fac19c1a538491094"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "a4d92c826cfae13d7a80653023254365415825a6fa9810dba593ec73f6560e0bc87fa9e5eef2e23076ad5214605be3d9"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "7fcf34ae19d002775dc86151d66094e11a187478577687c4ae76ce6e2c8666036d095c3a93c1b4332f6967a65f699f0a"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "e796373a85f5a0372d9d6ec70acde59b60174d4298718ef9b3eeb0222fc6c182f04706a2014f8608592493c3e7ecca4f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "1a93eac31a9802d78492522fa021b98711d9334434ff3c45d8f3b441e1805e08dfc62237b1152c4b392ebf85ac39a04a"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "55851677287615c9f5cdd01c36321a3e4ee0adc7aa8db055a7014b1eb8fe19b09d643ddbbbef874f8c80c6d1e30ab22e"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "3565d79d701a8b0dc31fb0d7eb545dd3f968a7e23b8a5bf09695abee2f1c8370e6d4098b234189c38bb0f3ba2869397f"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "ee6f7213ad92981ac82d431bea13e18259545c89b38dcd3a20906801e1abbcd324ba466f78385f84ba81daef5c967896"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "c966b502473611899aa80eebef4c206fb2e22af4386145cc1e272e2d3ecdebf1be7c21c03a47710957e82cfbeb290a9d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b2a7c12abd0703108eb404f4d038e82f1223f0aa15db525884b7d239e3bef60a5248a66f515dd61f8387e475b4a0ec42"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4184e0941a1fb27a613f91b6ffa7dd74ec65852fe63e108b94c8ceac88193620a2100d638bac1667072c40d05602c0fa"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "3a8838e486476a517a6a3f4bf10a0144296e631bcb1347f812f35f46f796977984dc7c1936c254f9ecd9469078a72ee5"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "7dedb138a250a3ff4325116aba68c7b45f21b6f83e1f381ad1df4e5b4606e5ad365728f3cc83a2badc8cd25bd45a4289"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "0b7356b6057490b828893565407e4dadd6e45d8e75399499f74ab90bc917d5b48b0b125ea3fc87f45ffa12e05aaa10e4"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b19abfc62aace1d49392cee8f69fc6477422a647223742cc7d65da06cdd8b86cb8198c3af9069dd4e8c3a4a6e404d9b7"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "a14ab54acad289474c703cd30f01ab844b8955e1e4a8764884b994c18c3f4812c0cfc6a2b762bd77b4e44f940ff4b245"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "511ee3ba7cca58960e5e3865e895995f988981d8cdf2a0fab5acbfbbaa0b2b5d92b3fd5c920f2529a5739fe85a8d94bc"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "b3701aacb50f918b8a79d8681f79dc8b6abbef0cf3386ee993f5a1567bede254794b028577dde83f45c67bdf9873c36052d04a2c03bbc931b0b90038d4e7e672a534025498ce19a527da09d6d268f59164a6a294775321c03a77af6d7252d901bf92e4c121c96be7ba1d4af4d642bb4728a5bfef56eac0beb999749e5e14042850cd6e7319b5282a86f806e2abe75daa273569e3a967ec52b38618739e17af10df4d6913887c969912d10447594123e1e8bf671877b2eea4b2a663933c4122e1f4799c487705fd59e6110dcce9365415f3a98238c8b30edd3d611ca73f7ca4c217bbd1bbec38a7958a21e91962b03a6dd15d82e8a5529325dd1f0d56b0af7edbd17ace14f2fddb73876760d5cc961b1209412fd21477c7a31848fc4996c2f955a91f5cfa8e3ce01b634d39a82fd0402b23bf509e200af0807944b228f99c5b5828104315caddffb6a93a3e3c9445f1e1a83eaff4cef357e3857bf79ae227ec0bd4b64b0eadc8806cde96353602df765c1f26127a281a44f5c6569247574af69738793c8d3fe6ab648cc8e741828516c310de42db3567fedec33b4dd863edbf24d9c2dd39891dc7c73a3a1c031b1f1a3e03d007bdfdc3cfc50493eef50bf890d941f7f7bb92561ebc71539c24d596caf7816328c98914fdd212c73e455954b1def78252ba637af66cb87b2e4db8d06e1311ff6fa6e044d12f743d057154e76a5c83c9b6df026f98ff971a9c26cdb64ebd3c0ca2fc0a399301b25f0b20049a037b6e36ebd2791f4c9c9698ac561b30b21b89f28a7510dd8b73559417fb7256c3623e2fb83ad0705a6b3e5e2786aa27e155801aefb7f91601aae443f3aaa501f43a7e1ee7c00d4c099516d2d45d83812494a9ea996acdce3cd1e7d665d4a584dc2ab36a3e0b3a9aa242a75d7d84ad28c3008dcb4f3795ec850ee6b436217bc8c35b677b86c45e426d997768e8a2300206c0b326dbc6585824896a452e3b2eb4f3f421b98692822bb2c31059de6015f934fda36103be1efa616cfaf114c7c62fb31e6a8c081ead3fb9e8fa99276663f497dee6e690c7efba83dc1f20b2704c234a01c95cd20fe765c92ebb918b5b22da758c031b498bd480d403603acedbc77701c4c691ae0e38852cc0dda8d2e75b95172e826af8afd4349a0945a731b33f132090e99362c3a98828c4e845c3b3150cc5edec3a63def8be1e1656cfa39ebddfef8cbbcab6d54154066362d228675827c7f1c8f253be4706a8501c1beb4447c3b5e1c39f74ed09d20ec4d5b6c7ab2cce64619efc90f7c9f480e21ef1d34fe8a7bcb6897862b207ea8710c85968259ae9e565f7bfe2328671edf2e4ba2164a5734a3496be4acd4585bb6ec423d580a960922759a31117876338a9f7df9e2d880fa88271daf2e5cb89f007507cd562621bb3cc4f10e44dd3b4aa2edec6125298631ecaa23222a39fe06f02f4cc2f74cb03907f5e6e257d724a89aa21d9457bdd57f331b230a4afcb708088387bac5deffe716610259a4e6d0f1d44497ffe4f60d471e1283b3fb057236ac1a5f114c5d3f59da9ec9d0c69cbe9b025dd1f077628c0de028cbaba4e49df77d4c2962f5681d15d717a3f427594c28874652326993d0583bb6717de32ab8eedea17d3c4b60078f9e64371061bd081ee5a6af22460a889a83540b973048417152a2d0e9d8d58d4b92fce9c1e03880e2aff73f4c872703695989f08416bdd6237bd8bb0808492cce475a32953d4c451848a0891dc789df2910b0c6c0f10bd579bcaba223642b447c4df4b7e6d368a324423b131261938a7e5b6104da69b2cde887a40cb8770c92f9f964d0534d3e499b1f4f73c70fe483cdac23213cfb333df48f5a7123ab6d66bbec131257050d96e1422611a80ced599b657fb265000d8422162766119bbe5e369898045e8587199b5d878331e32e8a6c19cf2eff12c25ef5ce9361fea28fd7fe816c1cf6591b502dc3cc2f014d2874b45ba9fd47cdc22ce40b46eb376a42727828f7c2aafa3ac91d176c9b72955bd7088d98a3eb7bad6c817e8a68785d86ac3801c3abe8e375f2fdbaf15f124648ffdd31f3c966c6e49c327b06702df6cce3b2ae781f3caf5d4c2440f2c07cea607da6eb73c4261a083a8ba32378c4b075b77c815dc312a84153352bd985fefc42cefa1c712ac5ff78d601696a18d92511a5fa2d1d0e0778d2a6b4ee2e1634f650624b478473a2d1b6ed283052a6d7cf412dc0041e76879b7195180a58d7ab5329e62eef1625695b52ee54ba75c95d1954bd4298da831f8abeccc2f2f260bead2fb8f995beb675d6dd683b5a3be6b59dfce3166de9ba3bc10abc6a4a73480b7ee2595fbdd6ad54eea6c930821d858dc77a2d9ff5c49278c8f5c2628363d40e22e255a5ca0680730d5fbfc8d4b33c52108cfb63c2e392332d009d40104cbb3333902c806cb6d67cbd588a88c475618f1385eae647df301747ab9b77738654174b8c4c64087ff5ce6931d7e4c3dc9bb14af56abc17000271b3575b9d4fb94526b52743b51fca84e4bb6c166175615145216bf949930e7ffefab44c15e74de4ee1b019118251efa5c0e77d4833b9c18d7315a25135e430efb7812eb85fccce9d9f7abb2f1baf5cca0ece580ce990558c3e5306c9ddb564613d623b841d2ae9a76a57627d643d7823fd125a9b81005a7b7c79f3e6d9cb0abc47f4740ee1b7b9d50def17e3a424ca19545c8bdb433357c1f52e718f704daa689edfe03ef29ac58d4eebe149258bdb15dd53f4b57751d4c6711b89aceb63cdcd0acc87fd9afc9538f7f3c9972d9b317d5ea09c11648159f2b8e842269237fa53526372311f05b23fceaed6b5057b56baa0ea8bd853ec6fad913ca5e3adeeb962404ba8c3cadca3089cc5108bf555d2356bdacaf08cba9d5fc6bfb8cc84b392629a2d319ce3aaf0fd28df7f02444e2b558618604357126c701d1a85c95fac0ebb29b016a2b29622853ca52c003c90eb81bb4b55b8fc1c63485ff514c38501b4d5b57d5ebf7ec9c0079ebf1e6e23439923d01bb236c2020a4f54f76ba03fdb0c18063b11da916b52ed17d45d35428ebea28e327ea7f3e9e9eec3c9944ad5629fede39338db31a3609e6f8cf23be19ae166bd35c1f2d5fb898b751397a78fbdf4ba152ddaa9a79973826e91e46a488f8ad725cf83ffb1607190337c7aa1b41594e865b0ff2e3b761201dc883ba9be94425d8da6cdb6ce402963c3ac471f8e9105e098c5af513462c683801c00df8cd18f60bf62a6ff76c23a6eed4c0608635ad1097913f3fadc84167023f33af6fb6ffc4881ac17c2ef0e2b177b04bd467c977fcacf43d21179058a308db5f1d551960e7d9c2c27e7824d84c8337b76c02171988d61fc557f540caa2b860b2d64d66d4f67d889de5677a7cee8df5a385e9a10a3b9f8e7832ab5c2ffcad69df78bd41d466efc6b44f716438fdf45480c082e9b28b65cab64fbe34f10734412563291cd7bfbd27c62b2a64a52142eeef8a6263ab0eb55f100da28da658f9a12534156a6777420613f7071e39cf161f30720ca17c2c2ab6160b0ca8b87879e47f5a06bd0c143afc5e317bd37f7dda8013e5e6257e6a225a28b63128bfdc1195644381872e1ce474e5791d36eea40c9bda7493625a4f38d5d89dd48024fa7d2ea48efa5b5607fafa4a2c4fbaffe838dd149c78ea7851ea9396304b41806a093a90aae59c0c5bdb170cc9a7d22b90cbce52cc1b105108942df20c29ef3a913223b915e7ebc98ef135adefaa0f0a6441ea05920e868ce9d1ff6c8fe4dbec06a4849e5e55ad0627f9e09dfcbad414a937aabd7ca7bf845c20a2764a00a9fd40ccd7b0f2d3525a9d41a26d7f7ec3c2f629aadaea9b2dcff61c91afa8f1dbe6c617978161c59e2f8564f58cf65a50adc17444fe7a4d43d84e0257695b5ce4410d114e9029c0d30d46d0548cdfd20d2d91c413989a22b425aa9e6e43af9655948c46575dc949837c781039edca7a4bdafd759129c8dfd140b22d40bf34329249da235c5abeb431e58fa1201014b7bf3e5b25383294c93da1fb5b694c77b63ce96d34c80778b560e1040fa1946d1eb9b0313f93336c735bd1cc95626bf8a5b7ebcbf724f485b192e75bb838fb2213dde916411cbcee62be58cc57a6ae156595d95cf6fda2743d3a711d4740b4d24b9e3180c64f2f691bc79f613eb0737c8a593ab690ac81b2bcea06cf1973420c0a6d32073ba51b8b031a1e31a71a6c9c8864786f70cc22dfb50420c7cb2b332de4ed52f01db8f1b2bd287da42d3ef9d3f46a2acdf3e6f649ba2b47bdb8e2f8c67f712cf1e8e5527eae75063f3db89bd132bcba78b6578c8f40c1f4178f3513441eab187329c4be5d5b82706954f1d312e7eeaff614c8461ecd5abe4b9a9c134566fdbdfa5ce185cd58d324c90f852ff7686293acbba3c386c477fee087726948ccdb860dfea6595d0ad99bc4d9ed131d58daee22f7647545da8c4e7cc26295acd4e6f9149b8d996af6185df320942cf32d9a17ef7494fbda541a1556c7e7190003829fb49021d695bf11e3ba8004bdf74b32b978afbf97c2e5a3fc25247a73d0dd40d24488ed816bba58b5a184d976b856374b45054ef98de3bc6b5e549726adb6a4496af25b3a3b6c4f5c4e8b6cc7a4c31a35db11cd9199f2b5db78d4c7b1e5442f5d9e48f561a369883ff82049315b165e01b151a5e0858742c4fb21769615fa349735194688222675e954cff93439c465574e417276cf15c5c05bc73e53be4d768ee5cfc92e4617aad0eb06e750cf51d27f286dbdc4d512c1561142897b5cc2479d26f8e7b47dd1999f0141ae078a432811d642e24b3622b7eec20fe26d6fdd3a4a5f113cbb37d8c4ac5e2cc7d9092cd897ce4d8a7a70ba3c02d49bd3ba7602b13f14a542c31291ae714814af45ba53667976fa00e3034da02acf72c7d2edd76eb3de11644c7c90c9866595cece1c6b444f66e4f0157c5196df2931495755dfa57122024bb8046aa8742d2cc0d2bde91482247ad2723a7a82986833d2a84c6017df76e453a61863cd93054610040d3139aa26482299cc0cf5df0316fe56102d1857433406974720ab9b847e26d298f8363fe2b96aecaf9d632b903726e16f7b29811d8deefc0f25c4d631318fbecf829cabf06b15fb63b04cf68630d6cab8feede57c3e7754def79bdb820cb5393f84cf86c5c9ae4676ca35c85b80a8726b0cc7ecd5d23a2244faa3240d7c21d257635148d941ae4e575f903996a2f68914493a0fc0e591f4cfc37d4819ec700ab592d47aa18f908393f4fea2be145b7f3328c6e6a3f5393cddc68ca05dea0a42c8b7ab94c4ffff76ab1ec39c384131b5562ff0da156d7a7b0d9843d540016b2b0f36491faa1f35980b21a19e9bd633dbebc33a829bdae03fcdad554f46af1d2407935f83d769fd8842c1fb226f4fbc53628726588daa04eb9c6a1386f86f9abe20533159c36c91a7d06442b018fa45ac68e026c48177430ef8731104dbbf6e6a07fa8119f493ac77328d1ac03f726a474d62ce6f0dbf453a704cd051470af2a82cfda13ce18ba886e7a0e5fb294494d4ae838fb098de65ce925bd3f4e0fed208fde1244b209110577e3426a99ced5fa50ac6ed21a9713f6bbd248e582ff7870cf9297cbd28506e5954eb7f53d0e91558d0b1694ab6043c727f3b71f39a818c14ad4b3d1b6f562d0d1a0b32928ba460609eb776845503d7734705217fa7295bb098dac7ee7bfed7032c27f2de13cbcb5cc36da776fcfcbafea3161f24dc5b60713a3cbf9ec4d6f6ac09c632af0fabb09dfd287e9b145b317521f2cdbba09444d7000586b4d05b0698316a648de5e0baa49a9c6e5931abe567320fbf8caa60834298a9bfb93517713841992ae31ce734112a1df939eb6b415561d15710404c765452bdbd245988e10ca8b6d8574b4bd7643a4191c59825dfb01ffc36ff18b4c88ccca6217eb41bb095d5680f42c77aff0e299bea84ce481b17eac4d1a17a45e44c74e76f30eb6c7a646601324e37f1258fa67145e0045e7ded50efb9900b5c4413da1608e355e7a8e8c936e3d2034ad5590f86467fc0e582b0c0c14153428472836aa680fe8a19b6a37d5d8666cfe4a90ae33f5ebc8a59c3f2d8de2b53963afa1ba51d59fafe7f4e150109bc74e45afb05b001111ed747bc43f7baba299d0c453d27ef16513984b9cdd2bfd894e71a55ea047fe01f2e1bd5f4bbdb803dcd54de78519028a9b1c6942e90e47d4f4eaf44284f005e45f5629ba583b510188dc1404dc0a5135d0dc167c79a1c0cfc91864030fe46ee2bd3a887d6eddbec4274cdf53dd39268f383e19f297ccb2a35464bd4a4909812293f5a250958b4673408d777c928fc9c05dab3c4b4a7763771d34c6be3b4886cd69ef3fd9d62d32365c07a42603f281efc78c45309b8a1c74adf14808ec8ac40ac88f27c338abca3702222df9374b9744fc47fd4890ed25c28b040ccf64b58402363f21b4c4ed569bba46dc9f368353e886462c3c3488c6883fd3d7bcacb20edc3d67f75481db1e8d0ddb77bd0880fd03edef9ab38c4347ef52c87477cb6aaac1656d6f4d775e6a338149c7feb62cddbcb7ac0e56a207ef1e4b4aa9a7c9c4a0d935ba5244ae5c81b5a0f89407466e3b3c6ee452cd7dba083db2c040f4afa7374b35b667e42fbeac3cf250a683c32d5eab99fe2cbf2546d8761094ff0884fdb840edc978d2016d33f0febac41832538f30c16465d4a391a8efec2daf8850ad6bdfe65c4bb92c505ef84c9cb86e9dc716f36d7dc5b5fe85bfd0cef3181f1b9f9851a7d5aa3fb94e747c645f482e0b713b1d25b1130a3e0967423e4e0173eebb87bdb89e5b0caf51bfa6cf6c3d00853946d67268197a6c04e66dc5a6332cd0e9804ca21d26c91640859c142a5e8b12077e45da0a9dae8e608578ea6377f3ca11ca26ecbcd51ef6020f86bb007578bac32c93e7ff684eaf048c8b36ed9370e31bae721ee22eb25881a58d4c58486e855598e721b04201d4d843d030ad4f4a23ef7459c1f6fb2e43a48fd95099a1167ca50a6a0f83f26f07c4f16c60097529927564025c620239c05314914f2a4618a69cca4439e142414fb0e08a4e6acbb73996c595778d0c7cd6767dddcf75ff1b0c201f9c2733f81fb584565083391b991f38e13c9adfe09f8df612e65ae196c7b41a539a62dfddb03f289ce686dc2b7a4c44fe1d4ec04849edb91bd477d6d1c6a873ac68714fb99df784fa75dfbdc300c9a365601a980a22ffa8e0cfafbc7abdf011732bdce9548de581b7251742add39d3b5e2e87cfefd3caaba9fe88fc575792deef0e2c17752910051d1db5d5b09cd0f630af1ccdbf0456abbd898cfc313a98c16a36e840e61980f9728641dd93036bbb250a61fb425cac8e65994325cf0e09a12ace98cb6d043d46ff8743c21faf75718368debf3f84d33ee95634dffb7cee6878fcc422d39350d20c4944eb1c49f60293f01c828024d3a191e7e7937b2b4efd9ce8f5acdf6dc087e1f4ccc25278f7559dcba2adcbe3b5a1a474e4a718c53148debcb2d15b3a954cc5af61ae88e57e52e920f8a6e96980256c27b4859aefa031869d84f90da4d70802a28d17883896e1be18a7d7a25f3c142f26de93c953a1a0dbb5f28cbf518fa5def6d765181844ea66db36a6bd6df2583699ea3b6d993f7d262c84f237e9c50aa2d3c701012d925527902d7d043aa9d8c9a960e44d697fb0af2ac2943e157f1f9560e6510a20870a4a5f15c33d6f697c307dc9f532696b21bed635d89b7dc7cc01c6bbbefd992bf2167174e031b33a2cb23fadd8e4440ab99b58642718c8a4f92c2ccfb1a13f87f3d5eed53ca8af25d0981492f42ba0645e8ed9d7ea7b9e1f8e5fbba890bdbba2f6cdfebd3f6a0b31502b2602b1cf2682f9de8117878da93e73e29bd0c41ab485e448c7b6513e6e6794b09d8477977d6b73d477ab68168e0b91fdf3cfb680d336a38c7447daed80cd8572a1a1ae4c4d27b2b8e41bf3def88ecef8eb2bcdb6c0bf1de12bf94240ea6e09b858dfbeb640bfa0d2eff092ac8dce131aad6e7086222738911452f1fc544f09c9edc9109dfc3a2469c1b1ac49f5d349ad5e1f09b6a277dee359dc4fec657a6eee86e40ac929a025c681e6a87ed9a556f729c7ff0fe7b6f61e1f1fc5d7c6f61e6f4cba150270fa8d325818f011e14d244134edba599aec688b91a2b9841fcb039466d29a0f794f43f36eceebcab4d299d2a719a25c62221c72d4ed94bc170204a5fcb4ffb9eba72f51ea900173fb3c9ff4abd41fdfe85ff91b1388bf4756a108e938e9aec86e0a3996f4e69ad25d931ded68a499acd3abe9cd74fca0ed85b3f9a940f80d04ac814908b6591eb183e0fa0605147889cc0ed4d11fcfa1764c3dbe2d3c6986344a1347f15570f5d6142b2087aaa9243f7b91a0703d8c29a952493fbd55c57f22f3c91b97fcb48809b97a76b881ed0596873b36eff6b89f4f40de64a1b29262944da9ad43d1ab3d141ad7e4e0af67835599157e9971dd6d30bd0515207d5b965300bee53cfcdf7dbede4952495f9f0746582560b5d782fa1472c53a83dba58327b50daccca6d8b1ceaed9a839b71ef72f6d75c58a4f30aec30626bb13f6fa91d714a0e839f71c3f0d318dcef9df6bbdd97f0d9361026ef10422308015f6e1282dc47e83cc16359143106fadd1000ebf0c37f3b48218c42ba16eec71d403e3b0ef6eaf63cd2cab0d37f2872b084290678f3d310e35b8bac46091617797438791e6b93c728460041db380db31e8cc781c2c38cf8bc20ad9240a349778a5e15533da4e2c9638dbca287d6276a8b7c74e45a9d0d3127037a8b467fe975d4703d5ff93d487b933c484965e3f5d76be5a9b4ccfae478a8967b0da3e98680fc5d2905bd8f455c2bb44bbce1ebf9a8545aee598667a32dd53c177ff58a9127567750db072569f991983f63a858b33bb94eab79136db9df2176e878c7eda9e767514d4672aa16d2695a8bd415b4b3302f237969f067a8fb11c9668e776d313fc36d96ee6c67dc55176dc984b67a69f41a9bd0c2aa0ff1002ac5fa07a083b79eef3ea62970e94b102064610c072458160ffa3a09aca1128cd0df4b9acfebbd933f46f38e1cc8b38218b0fac0944d0e0463d12db574d58e0fa887b3be41ffa3ab58191792bc60d62964e5ef58ca5eec5ddaa3143c9f17e3f71651dac8c32a1a45be277acdfc5c03849975c72b25d47370c902962597d284e1bec01960b00571f0d5e4b5964914cc122ae30abe333843ebb3521c06dbb0014de655080c951c5b4b33e320822f25a060ada042888f14de2d2e242e36f71ae99a31731f32eec926a36cb564281b0920de764fd72075d492c3adfb455c568dae5534bcd76a29b674a7c4adf8ee9929be0a3944510ca402123d9749e9ed9682b01d4494b918d24ec5a6e06f575c2621e8724b2048a37617dd098e92844113949f41059d4cc5750c8a4f446be7e74e6c20854066169b88a03395f857cd1d7a2ddc6edc390b068e7a430f751653167e7306d19d88533b4815e681792bfcd926c8f7f55cea6c428b127045d701016032c0fbe5d6f9873930f3465a6561cbea95f06282fc33f900076fa2a10ebe8b6b17f08010714c1aebcfc2e36f61ab92e8643ebc963ea829d56fdc4aaf60a1498d7a6c340ff0f286e9910f3e5ee25bea490e0e568d88b9872930f2e8af915bc44df6ae7822da8f63a91132c7a75c154a92ffe9d0f78eea8982074e0d400530a432f931cd07927539b0ccc80fb7534a3c4cb64bac52a5bdb9563f48190f20d241c31e90c31cbd3661872049872d5c7d27dccdbe593d61959f3c656b9c4b98b32574b345dad0bf558392c831be67dc2095c9b2bc67f64714939b064327a3c155d7de73bb74c21d55086a011b260546d72ea9e1c0494bdabd3f089e491b963f3cc156cc7c317f5bb7788377605e7440f32ae3e7548d7c70f219fb89f818a166ca9a742a028156feff5c65979bb8335f27430e30039526be458cea75da8f33dc919bbd18e8f4e78b8f1c86fe47c3c0d5c5ef9f683800c9274a306bde6aeffc05346da8059a0cbf6609317df7df21bb5961890ddc7c40d220528cd62013cb3874f588bb9c80293d32d053a130dc32715e8cb7d0294ae5e05e00faefe78fa197b087757e4a45eaef5686cb9715b976915999f49cbb2e606214d812ef9118f3882c5c9c567b55b254658f31d633117e79ef3bb7ba2e8130d6093a6d7bc95a41885cd795dabc3a2ffac47bfed5055eaab963c489beeffdd7ca8fca530adcbe0c26d69ada71896da44117bfbb1a090b9bde7bd6d3217c0ee5ae3cea6c1357dec3d20e18e2c17f5acfd5c3dcdf2baaa2c2377be53c785f0ebaff852d313bd716f91887d43730eaac03175d55d36086b"))
                    (acl2::hex-string-to-bytes! "be458f1d393877c0c12100466a1215955faf462b0de86618d8417af9a278316f9bdf2ebd1ea1d1169232f43d67d36942"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "033a5e38a60f821bfc35c8ef32c48453e4dc3fd7988bebd807f1d579df4360d488d097eac9b086ff1099a447a2ded1c2"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "e9bdfb8b069dddf2bb86059553f9b337e869aa0162ee8885cbe999dafe2185bb411aa8c06cc524e4f3b08dad667c6904"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "8246bf7b81b287411777df7ecb53a1795e54b150ff3dd584be2bc7b8437889b6c0e0f9b324e2a5d49269e0f111ac169885535c55dedb4b308f3b0f7841c8bbbfc956df2658a05ddbf2ea78df22e42e0debd2c85801eaaf543afd81f83f5a64a287f09b5b86b2575c2140cfd266c392df7a181ce1d232a8f6a6b946c0a3f72d65e1f552da14e13a6206b30c514511ea16186f8178b87adf20bdb78558d767be12beaf2f2c6cf5aef2898d58e52795ddbbaf1d1fa61b8cfcd9663e67494172beb0086457d8f4b65d169214da14741814d4266b1df2632ea62a899814194be0d648a65df56616ececd50992bb699462376c65a1a4e5f92985190f0bb3535c78ade180e22f5f395e0638d64d2269ef3bed885407db05434442b1dee06d7cdb03ec837e3a2d4ac4d55145e28e389eed6bde32dbe5ebddaa4985f7df5f691e324a3a3688af18af21197506139c248ff7b9b371e282fbf297fef5f5facfd0c0c812400386871816b7f21b8352c10fc473ee144e512734ce702d3bb2187ac9eb0526a5f1600b5408ee28ebc3884321c6cbd2a8cf101cbef89e9cff361fa67d26514d1209fe83f50c2e828cce9b6b811c652d18ff07a9c0f88a14e88af99c13f9144e2f52483722abdfc9a7ac7581584eabc70be31f1662579b5c9eb58ca4b4ecb6f2a3d4276e046c69c5e366fccdcde503682a853eaace9c383ecabfc20588889f0117978172c3d7353ee9021a960968d7d71dec4dae294fcb8014b18109b654384332a440898674a9fc608bb626345f8edd8b20356a9ef8bbeb2392b3f667d9e77f9d024afc75cc2ec6f106b7c3755290421d2adc2652207047a06a3f15c29d157fffb98913ba662d66447d07b6f4ca45ce7a112c2a704e525c32410daef02a2b0901d2e3157aca0ca200f78d47da4f15484add0cbd1e0bc9f18220050918088fe64b54424ce0813543cd7d31bd4f67687a3796632984de3dd0370e0f6d63bdabde84478dfe11120e115516db627a9ca62ae7451bb4cc4f16bffa141324f9d3e6b854b3ac74380501d33dca737c8475372102f21ba71be6bc877339f97401d2a73e65a186f67be3745330a1e154947a832731a34b4a9c2c97f04985a9a6a34058365c1805cb0233d157d4a40d32d334f3041c6293e474a80fc125000b0109a76069e421103df67d7fa8d1dae05d692aed7f6a36dbbfcee38dc4aeb308cf588e279fa3c38f927f4ae60ed04c401f7de4da61c3280b21d06377bf8da58656bb7e159f41066b13c4206e50790016be37eb2372d13e0b53a2b2377b69de6ee058fe923367f28a08ee8a7f3873004b83c0c62e7a3982c39d6d5833c0eb67a9b357960621bd90d5b9240d818c6754c5b640fd6541a6fc1c4659dbd555bb471e17e1ced932eae7346d8f33dc15fc026e9fde192f8ebeb3fc988dc584048086b4561e2f732e97ff906e5783f89bdb142d86c3a1f32b29af4dd9329da793e22d55b735403cdd6be1cf02a3853ee6ab91e9b2171d1bde90f8f5526feca8cb8b43e8814e899d070f94db729bc214bd15828c6c01160e709b97de4b6d39ee7420d054fd9393fc732a209a0669c1e3e9c1eefdfca91334a9f193ee0b530ce1564cdd83732ae5c2ea658ef605320a143356dd24aa389cf5503bf40fdc4aede938b5705c2a3b08583468e5d6b921260d607d58e9bed639b92f20c228047a656ee40394c76cef822419da2d21685b9c383bdd65347faa73e80c490620cc0afbd820df4eac7f6ddd9061636fc15b0bf4fc8b1f0fed36ddd1eab05dc8e17c23388fb5f1a8f3acdea003d3acb29dda57b7c017ea5bfc2b96dc27e3050e525ceca3fb6be0a9853b6a54e5385dfe9d2c676ceea8e75349a6fe1a086ce9e25bb48ec485095459487438809d581e5ba99b649bed06bdfebdef3d15f315b04be4cb30239359f7c1259f71ac95fb95d2a96d76cd25f915643e4d7a5551a40352d5eafb498f216ee804033c6134e96d6a85ebc488fe49978eb50ade608be289fd165917bddf2a64d7cee9c225e2e94c2c255d7757e0bdea830b6c90c00797eb569060ab07ed3c7f9469992b8307c5db1e720bb6bf331fb74b26bcb8fd2f8bd0652947dfb01f1f6729bf5b9e000b43c14a523caa4a5f35a5eff041809d6a85e131ca55dc3fdf9ac59db7774326572ca6b8ae3a7b7fccc85d7650450e2a1f4fc6c25cd3ca3c947641f93834437c5147346f6f4b50e6e29f86e82d0e8a2cb8af2620c509d12fa57b82442cdcc68a625e11777bb851080a84c94e71e51d299c6ea954ef3f17a17c7131867ab5f3d9f237e5d8bcc9a86caa3dceaf0312efab3354f865c0bdc71a8020f5c80a08f3f06a9c599a19c57a9b2c933c13cbdd2739e528bb5b22ca072019507711c41cee57305314c679f6f21bf64ebc82b18eed5c1ab4993efcb0ff3dafc85f69903b0b97ae24bf64c6a235767236f5059d6c6b9a3abe3b1735b5d9f8aa5154d856ab72374bd9027ae7126e33061290a377c0c67ead0f49edb722bad973b3fe67feb54cbf6e388412c48ec3db1cac346527d2298ddb6f6ac9693fcc6970b6061d70a1c1b8810bee33455d5cf51bca8a5195eaba512da546b23353372f0a21140394b7f8a9eb18e7e652c962096b353972aa50a151097b86b0fda823b4a9becf1f523f1a1a14281832d318fda074b5eacd4c4b0715d8b7d2cbaf84b77eb653e07b93804e936402291fb09354419d12456c642514260322d10c32301b1797a9ee1f3dc3bd9d4370c1b5d18290745013f156db24ef0bf0842c7a6cf47807b688a2113a508ebf193888fd1e2dd8c1f58197939791c14c09a58b1e4617f6d47ab8a45b706fea824e2328fe80db55b1f224de2c82b4b172b54c94fea8e3bc084a009b05c968f96d4af946c0ddd62fcbb0c46f2883d4e8045b43f2247b9ef32a902930152c82e20875828d4c1f648d979729f18f813a8d8a65504b98a591c2c3b3fc0a958006cedb3885a2a92cce979e101a0d69662fc98c3a3c48601004525a1b3ed28caf3ce46ff9cc30dfee5c776781616bbf19f9c49342ce9086a23eca7730ba2f03a06c4d4b211dddf74b55527b53c4b29882e0a526e5d0cc12458953a35937ca013bde30f7194151f82d035360ef0c8bfd892e95ce8b08a30ae02b1564017c4a8e68dab6fc480035ea7bac9fa0bb019df51629bb129c40be785b6d37458d5258b20fa707e1152131d61d9961c3216a9188111b407a803ec4639e97e2cb66819c76188e4a2eea6eecc3fa12bf4f88100ebe624a4b8e9fcb5ed490f13d80f4d1bde16bd7a86c2ead0fb9fc5ee1ce58bcbda0f1ff05d762ae34d21bc371eb9f0a64fcf1ca482248a741c00ac7f0c7f8344535a68195dc2fb1ed652654a594d0f338ce90ae53d8a2242be10c9314c3d24050866a1a24256f9a31959aed0204ed55e25f31cdf016a3ec7523eb0b74494daa83992310da3b8b32aa6b5dd6b0c260f809b7bd6d4865b3c6c68f8d7739e1d1398878bbacdea5e941feee5857d6568fa456df1ad7824c3a7bc62bdb1f8a13c6b4072b183659ebf38fd52773b89c49e42170de3253c754a0d41e01bd4a906895e4a7b824f6729610e3d81c97402b22c53601bceb283f034e9676191c427367a4911129c533522da706aed7aa38ab699cadae812fa011e4aa8c7579de4d5a995c82ed86e537203822c1006095e844a7d04c2136e501667c1d7dc42bedb46d842581e2d56b480540e8da76bd41981c62c9dc02e3f594655510300b372fbfd6b2685498203faf29cc6119b7f895905e25ef24f88e46a137f6e96ef1a38eb5baab8bc4b86e49b4b364f5486c1978173075aaeeac41c27003e712ff1e79f1e044f5f899eac8719ecc900c2d3c26d477badc6200c6035048f2b1f1ab3bc13f86991a159bd5359fb79bd90e92d02990f625e17f23edddd8732cdbf03af08162f0e4a24c9222bdbb4549c6633dcf2994dd990d691fc9d54337d8ed7573e4c341664b1304457868ad91fde073f6e22fc8743de688bc03f677dd1e6fea1fc08ec42ff6f7a47d803f5de3e0ada787b14214023af645c243455dac27eab6062bebd3950ed59602e249b3831ab765b7e36a60d1b8c506f830ccddee1be641f0cee056debb36419c458e98b8d286164d21af9427ded934277392e0e922e9f3a55078355324a10d135117f2cb30a02312121a984dd4bc3817b49f4ce7627d34ec5bb8583c729e622eafebab045d8ff2ce5fb2548bc3f678909afc0b2e06296c8f94aaa8e1ccb5a8b60712a9ab81dbed48af80b028d30d5e5acdb0030634760354a5550b895b6f47740095d608c30a7b4aab8a956e2b8b513cf6c31eb914ea610436c16a7e4753a72d437b18634ca6039afbdae9a3f4f8b800bd3e428ed00fee67eef7ae06b7ea31ac78cc59034437ef014678961055dafd6cc92859cc8d6d134476febd6ad3ba4bb846918031d243a647346a82fd77bb8749c3894abc768a366475de81f2c54d5e1855d175351e5490346eba97b18122be5e5269fda6b0f0c19094a0e1e5b22bd3954e94205751f115909c70bb769c980a751fd15ad266271a8e7b4bc48dc5da10f7d9e269df93a39d226692d892f0a4d354d1c84987beb5ab26d1e7143023e47f069dc438140bf090b3722ad54ea4ce81c41157b20ee024da223f7db817f7d187894341753fe263560b6b3f9af2e19e7c8d87857daff5fc11496c13988a67278f4a726208a83f8157f2f7da0fc6ce9d2d77da5a92b225c3b76201b49cd09b70010fd32ed9974bd4fc30be021c129452fa4ab3ded95f97033336a565ac33bc3344a10eadd0542ee3ab8c69bfa91d5a454ccdb91a3bfe7443ef8d0130b0f9f2f9c0c8cae7d6260f36cdd0a0199e3049ff8cc995159ed6e30c522ebc1c906b6449dd4bafe5d1571656cbc20a19e39389072007eed7ec300de5c4e37f3134d13bab03f466aeab2d6dca0b2b50c6608e41e5724c6d690b541a56ad72f27884739ea79a89974a4af1391af7d21a29fe699922e863bad2e58a31e776a4bcf86998b6c60ad3eac298eaa5cff6810f48a36bba45ec137c1a601a7a21854186bac98aab8fbb9a051d7b62d6706ffed72d38ddccd6e5235b4a664b1e8b4311f7c48e8fa48b16a69363dd9ff785cc121be0a607f65e3b5c0169de6745ae18dbb8209ef0c94081be3a9ca5ea2e9674858367d33161645586137fafb416106d6970d6e3f5433865aca2d1bb8ba32ccf1fb6cca9a335709d693c95c37bba3ba881c8ec43e53afb2b36740a07ca80ce45a85d170fa4990a7acc22b1010359b733ec686ca80ad50f3925584d70efea053ee71f4eec7eb5bb54786dde5196190a7472f75df9aa3dfef940f5a4e1410b5822a5a57b2a77b20a9436c6219b7a25bed5da1ec81e0e9a50d151f4163f090b4f59aa96a5b0d8b880a0ccd584c7f8a35bbcee84b4fe7e138b2b1efcab739012724bbe9bd566e9eea3e102410ebe58bc679374d2298341d0396e71b66c651b474a61a95c1234f17dfebfa2b2c766e2976cc135af3094a1fae10210ca60fdf5d25e01328064023a8d16a372d2b8a8de5df1aef9c4cf19e4b90c4fc3df902465dc369423b0d3b80a4575f33c46e86d93357fc288b810fb533441bd549d02c0b28d5b834293683eaacda219c6b95f9b4a489d65a934fb20369408b489aec0ba250c543558d22a7c624e257bd9d1261b4bbdd3ff26e09771d500e99c2cb7d2fb0682a9112c14225b79f087833c3644d6debda9f8c563ecd677a16ddd03633e6a9d16e381880256c139da4f97b991c1b34b70a8b84f660238669f1bdfd7570eee31f189eb3243d097e8350521168aa699d9fd57a78b90e27de8217c0a6f3e4a710b3a23b9f38098c6cf2831881cc8fa72d18d3e2e438a938f464f16c795bef8da741a3a529361b43536dc5c3493f4c5257b47e6b33dfb1d54c852747c0f7f2211b2b9551eda8d2ecb64780d8d43b5109a1f54d6a0428c89b6b5521b93a62acffd0d3efc8590ad0e72a325cdfbf3b4e2b905db965b1ae5de9c5776957222670888cfd1b7ef7c685a11390b23e66d99436a7505af4e8dad24edf4e768834bddd4993e9c0fc467671c8f6f5197b6770b3b95b6faa8df638c6393d5cd1f2ce9353b19a4c48b6c19525bd68d20b1d3dab2d32211e10978632ca475424f3fd7c31745ee4774b4505ad020cddb9ffaa55e3712f9cee25b805732ee7b829552eba229fb0170d210d6f289725ca1f7f995ca4201023527255f05494a8ddfc8e0d04e16643a66062967a826596b264f3609e87aa6268dd5ef7c3517bf939f106fed8f8488817339876c454fdde2b3d03a6a7e880b4b20174241b14455750b9489ac2d1c70bfdd93b04b92ab07a990a18c215513385a3413a4154079141b6b7db574ef8523daed3b2e6e54e3dea905ecb59435c4333c12f1a060a95e056e07f27cae9e67a9bceb5a893b569e6ac85ad63b2de54306d702dabb8bb59a2f54261c64c0026c0b4b1f17471c5f85b4efd689a8532f37b9154bf67ffc2cd69bb61223f713e33a5071d09bf2783640c307c22d836dd94952dd376c731cd6b94f7d112ebd0f09798015480d11b6a476a0f666e955e5ac20db160c651f537aec558f6c653d1c3cd2a41fcd962d4071cf51a60ae6a52270655baa13080c0ae216b5d66bf1ffa343ecc376a85ecd7d65f5e0fdf64cfb8cedbfdb80947ec1a7d4e0753955fa8b0ae771e504218c79d5478d54e57142df8bead76280d7eec69419f19f9f74a76abbf00612b07f49bab0e142b645c5f0da873154f8922381a43779f33af20e21231c265553620cbc1676f9b90d04387f73e888dfa59ec25ee2fabb8b4f196c903e975006af020e0e178f589d610c13173316b820728cfaaf0844a963e5fcd61b439338c12f79ce664564c694bb449edfa17a89be664414999f7c0923048393eabb78a42ddaad94ed275b0b4622942000457ac2628aaf57462e0ca473b496ed262ead5df2b887a8ca45ddee5d9ee4e3a2a77f2db3e4b30dde3d5b42f9fe1d2af7229c2bb9d1ef51262753387e371c2deb50f2905a7f397c8854d1ab60e900938421acd52a2eced8295623b153f6254706c739b4fe602aee6dfcd6a64e7e2770f6b24525ecd60a85fa59b72184e2e5791994be6148916ca9b5bcd2fc6cc47a09745ecfe1a8d18da0c2b87550b903450a4d40e75edf19e9f6871d2d7561c3e2a527abb2d205283d71a8fcc70699622d44c9521e756d00871c0ba64fd9d583458e372357c0ace20440d5ac140acc06866db652ca6c974e30656bdfa1f0720d8d0acb4e24734407c9547150c325fbf836ce3bc8520c070ccb17cc508982dd68a0620bb382b276a074f6a3703e1fce3e27af813d201d6542407b59827056c19c68de20dada46eab3ec6e9395ecc0ece23f58af399b6758f677d937fcbf35c4b2cbc5324b92f8474b1b046a4635cffeeb6810ee422bb1893257799146dce78eb1b361688f148e356de800500c8116ab62892066e238c5a0fdcaa28de93134c8a107f8eb2dc6d05e233ba96c6e385b9c4e8ad03ea02a694460b89d536c24e41a8aafd3e3ad12886774fc48e7f4c26778b752689f046a661d6107b30c571fbe2b75923a77985f7cfb691325e8268802e8a6cb30b0975e84d40ba029e033898ba4699a74cf741475c0086988988df0f39421dc709ae06f8a11636b4af686ca132c6faecb14cc9b3b5e5be4b91fb0630423336d6f846805427b73505d77db4a1e6c8eb6856e255b967e6a6dffa4174afb3ee1677697b58972bb4238a98ab770740cce9beb0409bc4ec83beda44f4b0ba18c35b0cdfa5161e379d8097758a4a6dcb3374345ffff262ae3d94aee49d310beb63631bcd6c4f75d5c3ead0f0929542009d297db869001f478496e26ba37bf1a63ecd28b5394b2a80c7b0ab9c4cfe006b8a67ff4c242b5722e943f93ca00827c3c607c81984bf98b32970abdb11cb35765bbe4dadba0aa6daec3a77f9fb5c0aee89915e7ce7036cf769b9462f99a35daba1aacd3f82f49ac9fde8d3064dee076109f86051c8cd03c7f8e8ff2c4d1ce595fcac1284179f8f9aaebd8ef3f044c9d5c7bb823ab262f9af678e5b33948e79afcd30921df05201a77f4720458ae6beb73349289a3c6bbb182617dee8187e667383e6e05fa5b2ad8ad0042b7d864a527fed2fb7f2bf5dc49cc523feac3163586bb12d71eb9047cfbf4183050e7ae621ba45edd4c6b39fa297b5213f1287e3abf2e572bb24587804bf6432416ac150a8d0edf1f398c1d7d1a386ae149d8465fdc3916540ac66c9d905889065250200546e45da98eb626cde4aba1913bfaf60ba38356396e633c3a91c5f2be4dfa58bc0ecde1639310de524f7bff0b0fb716a5deb51fc225447ec117e387354cd34395ac7171358fc3cc7423518c370d362f97537679d77992ba99ea5a4f519a036bf9ea791807ccf6789c27d451fe7da0c848cb6f6d60b075782ad2d352fb7eb158518e8ddbc190d26b3024ad5546a61c6a200d480245131441dc97f6991c3f0690036154c44b247f2c9ae283a0f9d1d39c87e3a8416eea989c17006a2b9a26299d792b77da8dd024434029ff3bfd46ece6e5133730c47d281243d6d8115f7b420719b49d256fdbea398cc1c92d3c4d90d0a2669eda2d13ed2ac8e1e75151867b544dc36faeda8ff895e471887e9c2ee7e326a4854492f60f6c0ee940f55a8efa0e7c22957cb43277dfb202e437073018a6da959378dc6cb5607eb68a0461dde7bc0d97b697e70d8ea05d3d875f649c4ca22400717df1be078f7e0f6e874db96be76aeb7a085968665e4de716fe9bbaaa5d6dafc75bbb342e8077be9485086bae16fae68bd6f48b26a4eacb3c1d93d25b611e683c1a211552bbd289a306793ad7d1254f8e099ebdf7f79cb09fc2362c86369df478c30aab9ac9f539441388700ea153f02eaa414701e1cb5de7f86b105462541b245e603c2f314a961eb21d547d0cec8f11c0c4716be9df9bbdc41ac5ed6fa9a4e8f10a11397b3767daee202d499c58f5df530f835e984032c9302b9478e506596b86f072df3026761fcc79449dd900ea416c0ac5017456eab24387c51d4ad2005624faa649d77e89965611e19f4a4a35de119d7f7d078a8b200514a00628756250d410f7a0f8a769e693ce4b77779fa0604db0f8f9d942dcbd7fbb6c58065747029d1aab0c1b09deb88b99cc4ee868f16c252b799e70e7a62eb865733e6de0771387000bb4c90cafb4475baa4c260bf1ada3f64a9356e2b11eb8f8fbdee478b92c9aaa5fc18918a681cefa2e98b8f268c8cef9b3011399c533885e4d60f387d6f5df00ef5ae49589c82860a05b26831e68ae1dfc102d76aa152f7db1ed227597155deb02a861eb21305e7348bcf928da334c486ce72ee0276db53f6eadfaaac1351c51e2d388823662771d6de4b84866d1af634caf9e3f06e3238acec7028e1ffc251cd502e1c290057b4459faeb4a88ebdf57e48f03fa0d1e80481763c28978137f32245df514f6c273d252271a980929e50a7cb0e77b05c7d46092abc3049321327d170d4bde314166aea193ce99b032c8665c3ad129b58528ba87c58c6539cf47e3f53a6b890a295cc08e658eb547af9052cc544a6ce701833e3ed9a61632c5c54e080bde7e46235df060c6e354944746b51326d9ac61e3edd4fe10977d46aab4a596a92b24b0d6722661dd54de61a3f1797ad90651ecd26e641191e9043d271dd0e83cdae20feba24ad7d369bb746a9985499559c350760fd6bd852312dee307b646eb74222a09f6440bcfaa54954546c1c8815b6b5578d7124b14ce0ef2877a41f7de804bcad974fc45faa00f8edc01153ec693afc380cf000365716241ba7e58453e86c5b702265bcd7bd25526d6d169f58b89f86135fd892ca1947593251ce376330ef7b92d1447ea7bc88f24dcbfa533f9c6aff8406b930fefc0afb06f5bcbd3e4a14b980245a9e5220b235195d2b14138d13a50482107f5787b78604144f6a47ac6281b28c16a0697227b75aa1275676f320331f625ce246450386a43dd4d311c06f60c489070950395fd58c287daecc7727063f281cee5dac457971c30b8c1f3e81e3109bba5da8ded13c1863ac61a6718ebade33df17f02613daf7545209e27f406521448f01d5eb124799d32223777acdbd9725f1e3c05ae537af5226b0edfb21739104238a59d699749b177d78c21b7a8ad46f13d620b33ffbf45d1835a43abb9ada6ae67bb739ed6f76712cc618bc0b9f208fa353a3b79aa480c5a4eca7c6655757e9664a708d6484b690ae8fedd4f786f5f83f00cbe07bddbf3c3b6a5b26b515a3f0117b1839c550f5f6715aa40ec4ceef4935520bc659e41a216a2350c43172492f868210d756509f0323aaedc209d356e324cbd5c1cb742c05bf9c0b3750d9b1e823f3ecdebe002c5723e52d872d40e7668bd2cc6b36fa5f598a58fcf899d868ca78451ec852fc3862f0bde5c6b573fb43e90b623b22d34ebd78dea87082eaf836f1fa291ccb811da71889a92918f90cfbbada19ba25bb5471f9918037927dcace3f879e546e4b769419dcea06fe4cb70e8fd35550a60f1b479b1636c64f2d6af0af81e107d1b7bdca632c1ae8abfb63ecb66bc7a72a4b0d8ebbd11ea51f66533ed05d839f9c627dba92fbce56c861be26fd17c31628fb95b80a56ba4c99b50e09208f188404b810d517c076c9ca3c003d927bea36389d6e63d51b9c35349615f03eaaf26dc14521ba602ea6ca27c6d4a134ecaf7fcfacd212caa436e78685e5848915b3b558761acb0a7ad0d077bec5e2430e856b64a67b3549650cebf"))
                    (acl2::hex-string-to-bytes! "4d7909d8e5613249cae1a82481093c7b3260abe835294e5f72c6683a65bb0cc5142e6b8a21beeae531e7c3fc8238349d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "eccbebc347859d4c096ba63c585986dee7899de826fcf7fdb61c6976209fdd62c6aeeacb1a47a83c0de0c27fcb4dccaba5e3d75f4ede43484b00d5e9a57b099be111aad2317a9fa3ec0608833c838a521761e6987014d10eaeba00605dfe8231233c525ff6e62d898d26a1cf14c467d9244bf21bf2674857c7921163b0efed2c76760bfbd052f8f7d887b4ecbcd80b4ab6e822713e9f512b7ecf307a0618f595c109fd2d2787a8f4b5736a609a871facb4819f21ba849393bbbb872ec73f4073ec3b4f7d644ee6890c30ebb518994b08055e91b3d0ddb796669c3b065b2d76cede552a73ed6d5abdc5075e729dd5366593c71fafe774643d67bb778b2aaa40ae819eb5982bacab98c840eb20900489cb4504719e660f438fa66061439d7e461cbc60197386d58122fd23bc465beeeb43035cb008b777fb940bd321b81a702f1f77e17d8d67ae86644df8f66c655b05d3b5190473417435547f5be0720a49a3c09a6b0a6c340160659e06c41b16137f86438194696cce99d8e841c6f8d6d14c060e44e79a5794b72dda964cd9d6dd006196dd8971a04cce2730cefad1e4d0cff9b9b43e9f362f34c5e7f0b6310d83306e94740a1445b77d493d47a5889e0ed7a1840534013b9c7e34af5c2ba823cbb8b4269040d8c3d74d47f9f7eea26b520a07577df34acca23e821822d3c8807e403b357753488374e7284a25fea9b26610321b66a30153081cb80c89d5188d646f0ca7726ae9eb56f2f41a18fc25dd10922ab6066baf3797bfdd72da5920b044ee2e66c9ad45644417b35019119c531938f812a674dbe017d0f9315e896952e764a958c445e35708fe6ed47065987c49a9a3168de60db650f615335a8b7adc167a8667ab5a4b2a684e2d8e651fa0c0678e33a905e8ce2109da375f1987dbb497d8cc7ac51306472db0f66640683d873d8eb1b554a2a4890c246a9de2a61544322583785b999b141fcbd3444867f53991522421c33d1a561f7bb3d5fc0faff3b10ab34bb51492fb0548cb10c645533e2c8194d59ae1b5f00b871cefb0ec3128da2e65da1bab7462a8cee6b20229dcfed6c409b821b5ffbf97f1acd44662244d773aa10dd586c7b3d1062d12443c1ae256c38cb4a0e27f2c8d616b6a069b0fc94ed56dcc2ace16554697a718d4a59450762524b05e8fbd9db37e629a37bf7a0428a8d3ff64abe6b0d02b5dadf98dd2b1fc93da95a4e5427bd3f1984d23f5ed66cc5ada8ec944c12c366755b7fd1c5d6eb1b4b549ba0c952fb36115691066f15704948a61a2c22e3142f5991f2679bced5d31bdb5e82b3e1f1c77e2d7473a0d3f96feec61364cb627ba7d31cebdcd7798c8b1fac1766bc6b5d5db06801098ca864b3f0774cf0bdf7667a623ab093f17d6b9dd7556abf3b458a834894bb2a8af94eb00e483b32734861119799d54f8230efa5d355ad0ec906f5b2b741f779444d13ee889772bc73ac07f7e3b8ee42023aa51d842b72d7de364b767cee6a58111a60ba000367f137b8a7b7c6f6503de9f1187f6a5530acf339e84919f8785999f791ccf51aaf7347e4dcbbb1f0af75a20ab64606518af090b858a50f0ea5f0475b3371e82b9db52cb7238e4526408bd2de0afbe8aec6354f081904dc8f77bea1bbd25359d4d7294eb5f8a9f115d802f4f1832b71db970d86cfa602f2f9620721afdaebba94d0523657e899363dca5d70dbbb3f424661cbc0658c09f2daf98c4f9dd9959afa1dc27c136fd6950cac1deef31fb1daadbb02947037945cb2a290940c4b0b6b6313fc788eeadc8e00d1a6b7de2d2d9b4007346b56453c810ef1c69781ab97bc6e5762b411062591c8ef8a74037ffd04cbdaf3cdae115d105e6b90b28467d4fe7a5c87c7403e4c40b35a5babf964cab79faaa323ee38907b0f84718d8b2c4db22597fe2a92ffbb0b5eb68cb82687f12449fae5167d375131b0b10b710c556c67ee9e18469fc6fc523968802121661acb39401c0df17a29c4e085a1159e7fab9043d378b3bd207ac5c1f4e88f606ba22d1c1eba82e19d84b4fd86e4e3a807a5cbb3dc81ea8198c0d686f0285b6a2521ba30eb631da3fdec8286194953cf89c56abd2de663739c78f506dec20db51f0403364a8826df37ebae18d130ee5b82950dce2bf8b709a7ddb1b005382ec7a5e3988a90bc46f9c7ec9804335e2c026e8f15756b4b4dba47999d11281d3718762adaac7251e66a1a79e0f6b403952aba3e22317050901c082707b8abc3af3dfc545a2a43fb9dc6cc1e195ec7d3a25b020ca480cfab369545f08204f4ddff9ea4ea8772aa0bd5864943a1bb0340ec09b2852cf1cdbda9bcad90ac84b7ceb0b10c77e313f925163f1e74fa89eaddcabd58322bf31efa52088b81e7b659086b1b9f671d9cd5b9d847b84cb449e2fcd6fc7f9e120327f3d7ff8c4eb4285d3d7744da52775bb44ca436a3154f798030993d85237fa06d3ca4080c41a24938b07db349a4810f195002827b792ca7472b79f6f55ce56973373ec8a82522a78013d6fd69e7daf579666bc18de2fc41ac2001bf6cc4b3bcce7600a7dc43c5b1a7da2b69c786e5cd58f2586a66dd0ee733b0b90d47d59b28d36ba22c15380bf62465e3aecf2c21e53c7640e392e1f4e6527018671309ec2c26aa9bcbef7c55e38a698832fd221feb9874cdd3ae7c4efd9cc098335afc27353c95461dc77b34185e37917cee13502ef17057a0f6167d6b8744c1188c16ecd299ce498839df36f2f4628802fe7a334ddbfb66d2264af0de865f1b7d22d4ad92a2a80ea1b2f2df02017cd46bcac169c6cc7883fdab8449b53e721fe7bbcfd9e3fbd51de0cf608cd3e9e92efdcf734418276a2779e24cf88df44046ceb363024519b7ffcbeba2d0d73239d05f691a52b08152c9dd871f8dc76c2c18b8a638a74460d31cc994344b8a6ed161074431262e7ec503aef6766a0ffab81e3c10ab495085b8fd70658f4d7bf452696bdd323ef054334d54bcec2dc62456bfaf6f5558f7dcdc2a46bc3e570a573f52139097fd94c6200f6546eb39687d9a85a92ea166b8fabf2bb314ac08e42bfaa3850afcf28a390dd25720fe2d856ae5f8bfeb29f30253c46a5c1b62d367d706d646a8bbae4f117f3eb0d78f6a9b70eb913a11597e626c9e7e981ced3723ed0682bef67e916203d69cf8d729812923fde7ce2daa6967c9afd0fdb5c5b496d25c018470903c6174b675a5ac57019cf8af54e2bcb75470ef7afa036266cc4f3b8a0b0da39b1fabe881712dc3925c95637260fe3aeac01c4282523faf6afe9b6d12e7d11c19bbcfd3e9f817b9560270f8e9b49714cf3cc867488208e92ccc29fb8920652686813bcb3272fa0a3c83de2d779833f6df0b13c25001f5cc26a5796b4f5d306f5c108e331d8a3b1290612b859eb46b4d84eb7ce81b4a9cb5031ae840e7f6d96314da15afed96518f0c1ae775d17193857b3e724d1cd454302cbd14f0f575b68e4b5e8096253fd3bc0a21a092b258433b0c847275c727c2ca2ddc8e1fca625e6d5a0e5d9a76ef48461515d540b3e0f0dac34e2bd6cccd5eff612992020935477ceb79fb01351d05a9a757f42458a432f139fbdcf4703d8f2c70f4ef6c2476f8295a21e78f0251c2434b689ba2030a7ffadb3e246363e0d9edc7fef4a29e9fe71f4086a1784115d4cf8be82a3e558c4e64215580881a5fd5bff0882039ea289abb0918c77183e1d39ab3272e7dd2c204d997027ff70612f92f652110b699a999894e68b7ef3d4e35318bd616d4dd1459a7b6a58114b9a2ba98a06a349d2f8b5a067af6f165ffa38020605581d8072bb37b0abf12859777a0a83cc41a521e5662bf0934026a2d8a3ae3cb25edfbd35088a00dcfd2be321a31ba94bf2079d8833a1f83b07950597acbf836d7c9b4b42b6fd33a5810231c02e46c42cbe578f0aa46a474fcc4dd05fdb9da3deb93973c24df610ad87958b7ba4f29b16ad7d3f9241eea57695354b57b9b064ac1da17ddd53df609d16cc397a8c86f7b2c862f5499396da586c35bea8cd618e7fb0bcb1ba521feb1c378f82e66e6ca0a816ecf95621c8ad9e583570799f16e8017b705682d11e79d3f1c6b0e7d20b5e9f9f0492e273ede8f0621daaf3b8383bd885a6b65c5944d26a4c5909a441cbf9406c930a15126e51f590d5b43e9f304facd647177171f9bd885030d64643143832847fa71685b1b156ec46d681dc65ec267739c2e9a1a504d506335ef45e4ee41df0b2b20e5420cfbb7e5658e5f83edaed6f66ffeb71b06dfb754bebd71ce1c5abb6127224082275ca4dee81917a25e0aac6b4169f6a82c19fa55ec3170436a382e0893181281e692e61f79aadea81c805850d4e2e1383c76c9f552fdb9eb3464a75fe354c3c7e748dc257eb63d1b958a5b358f0c0f41c49076a19774b73c176ac7e58c8aecda95e9079f8bcd9b6375933587d447c96b30d1b0e91e3900ac9fa4f028130cf6207efde8cc05537216292567d63928b80b2dd4e94a4bbd2eb319bdd0960b5d68300ef200699fda69aa2cdb728f422153a7f7f48033c46389f6221fb9cdda1ecb8fc25fdec6afe4eaa5fd0fe64173b414d84bbb26340cdebadc55453c1e03a388a635a1e481b9e56d80164d236b4273740275b34bd9798139dc1fb8d8251cae3809d562a219326e6adf051dba8b42c209bd22575182befd6f8bde051e71897142312e1ca80bcde28b0955fc2c1fd9324e7ece31e7ce31a5f4523c945f71eeaf4f92bc3a66c7d0d426956562f8978a46aabdd5bd6f1a760d40c5a7f14ba490aab8c4ec365edbefc5c609cfdc9090da61b6f0c5646805f0b5ed6caca643882416064e4abaf0b67f49f1ed01d3948a5e35ebce39b1630f3612429db71de7ab2d53b0756412ecf613b322a587e264e26a15c18c9f4ed0a4cab4f90f53e2987bf0f65ff0a4e874092193e0c9ee839247a293c0e262773145030069080f7ed67933706d46a996567e3d8eb4d08231ad187c3272363fd5158e83a72b3e465d4231495450cddec6e4a3cbf607400f6705d37b645e1e63e2fdefd0fba80895084c22c704b31ca0d4af38615c609ed2be4d1704cbdcf18b2b4ac75f8087fd880a54cc21240ca9bc5a26aa723449437881d341ccbb875612089cc3a3d813937bce0cabdc14b60b7dc1eb58931039e45d6c48e227c5d5c2a52564632264f350fcddd4307c6947a861a65de12862bae81dc80e86aa7d50bd90ec22047fb9afbc97c43bf876944a0704720ed2c6ce34d5264fa49dc94470ad81d756460038914cb8fba4625f1835bdd2af6e8be2953022034d9b9608adce28d47181a648745077fceece1c9a87831358c90d9b006a4555416c640ded3bc1a625353483dcff65c4c037e975bd44cace03595c7ea4bfece54cf69e73eb50e3abd0bcdf8d223c29c99f41b71c3be265446edaebb534088f6892a69b56879fbcb08352f310bcbd44ed06a6975e9ae2fb0d07c83bcd52d6b545bee6fb34466aa54aa9a20fcd00e52fa582d10adb6eb8f48e4a9bbd25759ce7133e222e7bbc6435d0b024960093665299905b92110bf6da4ec3dff05c20ddad377d02b5bb3dff06e933551659e893461fbba0cd8e170f14d4e01f65efc91b704917e311715fbd710aafae2137750e82de613b0da0f4745ca74ba49e4ee2d8c8ce7b4dd21ca6020cab81aa1737dd2a812c2da4e2f838c8ea3c4339fa0802981f1e33cd783b956f425255e388c96599d9af7d58de02687f761a833a1989bccc0bdebddc645c9277ba8598b4f66ab2bee4cedcd16f3df80a58b8781f7d6255f6e77695e526de419dcdbbd6c8d0aa591c4b00c89907395bec3a9c8e64862082ab6a8c4882405f1764046b7c89b7f6ad3474fbdd3a465328a7743d52a3c422a75bed76423e414d784ce72b06620cde4399eb06671fad39d2ce9e215157c807588ae8cb1f73aa3808cafe6110fc5fdf90bf25c712c39c4beec430df06eb024cfaaaf5013cbccb9d3d50a2a408b321693906ad944cc09417d11aa4dceff80342225e9a9f268455eb8a908c7a7a20e2099112fb149191451c2e54829178e92ba553f8322b6974b9a174ed144ac1c0224f3d6686be6dbfffc75ae452933e491bd5041a540c71e8676d80d8c29b9c0b52889273e9085d9754a7eb5c9d58bc92346fb73798d8dc5a0f9e42e61ff3557d037434cd9a78ddc1c44c542fbb68505f05c3e1c249071563e2c21f9794c88a9fa56098c603866ac9cf1339d03538a5fa1070185d1d1d9970b1578b6fc9b197edc8fc331fccc77119e5aae0d0a3403a0de4f2cf7a6d92e52acbed1ac1271529e1de95981d4ea2ad3ca851fdb38e256904e3da7b8cb2fd94dcc2f1524e1589ad1666db3c5111ae6b9c77c6f50b4e4edf79ad85a10695b4051116f2a6694d759de5f21b944dae4148c3e4784b2a7c22e9d6a292d49a8e6cd252385faba5177f88eaf0f544858d11a14d6ec48493805834a643609aaaf57e793ab7c6c6840bedda9ff3f6a17b26861193875a25f903453c53309d47aa736f561515967b78b3671f7f6b7e4fa113151630be9793ad6d705d77daa7802b70c6158bc26d5f157671c3f3d81ab399db552b9f851b72333770348eb1fdb8a085f924095eb9d5ccfd8474b7ba5a61c7d7bcde5a7b44362cf9bb4e02472e759c2baf0e183aea740181158e15929bcebda0c28adbf1a053fd1cb8066d1ab82b90ca2e3022f79dbe75d9ab9ca7f17d7eddbeecddae98ce502618fcfa499d096f00ea1d36d40532198c4fcd0ce9476de6ba9ab2234cc10cbf2a970096025eb0a47f13e28e85f45619a75be07b4414fb8ad6bcb918211b19f58cef596631c03c606dd024a3c345583369968c5462da94ffb4ad4818e5f74497cfba7df752721b275f973a231668d6fb45882714a008c8a9d6e505a1e5f63b1a6560c4a5dee4cb6dc6e1b968e9ef50d5cd9957589640151dafec683fbb885c01e07ae40e5188cba8263828f1d83944675452838cbff0e8cfc23bf66ed13e25c2b37152ad34ee9a6b0012ef6a4be720049c3c422d779be234904c26abb18836e170ec76ef0376c62ae70a931ecf697241e7cf09b777b16ff0aca57f72f6e6fdf28b34e76047c1cd6c4d77baabdd40c9b3d98340b71ad2006869c06750fbeae95a74f32a8d7c402eb3a5e2ac7de00d84bee3e18da8eb6e32107a3584c24af497adc09f96e30b82487572f40f21921031e79962570963f6739d306d4e536ea47750e0a2d2fd9ff8c93646f459234df96fb25c9e23651efb53433ee03a60d16b06d86795552984242f8a949bf0ef8e498c93bffd892876fd4fc6f836bf870dc8d4c8d7ffa4136294c9c6ee338d02dd51172035f8dedff66cc77cfec964428e4e5f96944a3637b5bb469fb65373f9b5d09ddc5128bb0827da68f9b520cf0fed5c05c62efd8202e81824697f845c3092289f2c9405a41d85c692b1adc5366b758265dc2a922e01942bc1a8cc4e9c441cc9899560f800ac9aa1d3295b3f50c1e1420e4cf99070b02476ccf20f0abaa09043fbab6aa87711d0fca8e54fe7a4cfab2b5d9d3e81109b238994c9ca413de2a72853d2f37f7e09b3579269fe6688d12d9ee2414bdc843f1bc4c79480aaac66cbd720a0adf56be0cc132e2be9349362f32d1addb41b31f867b8c3e6b47f7cb968d0e0a8fc92e73385efbaaf82ca4b11964695d6f9b5a063b1e61834e868255728c6223051e0667e7b2de6697da02659818bb1a1af44561d64dd3303aedd4cdc660753eb6056702e165e2fee1c4eaba74f6d1cd2274fd9620afeb2d75a474995bb14f44e8a7d8aced200f5a1f13e41874a5b25fd042090fe1edb06247d4ea23f928f91be42be1cdd58b6e6398ee78e762ce66b66c87f132e1ae2bb8faa5bfd2676c7aadf2ba0acd5f4d06ff967e2f8f0418f6bc3e6a3e69e79b6215d53e7a0d6f26dae0597cdc14540ee0175fea9078222c69578d14771bbd5c5f631fc8a33ed7fe229e6398de0a7b7802943c5b240b86b1c771a3ad24af29982a88216ac3cc689b59ce563799179e31765c75f971aef75018f1c65ccd8ff4f7c96e234d611feaa687587b8093a483e1445952eb01ca3cb165bf3f49fb22e62346fb6c98059f25c6b4231a9c5617f33172711b5a5cc81e7bb5c3569d35d0c6c1a819467437b4e1a4e9197bb26adab6c80a8e176d080e655711e5528004015e7cf8aff70980c6e5e82e21e5eb5553945013978f4d8f06d9ec800d7d19ef1d74f93da4c87e673dfa398923a26fa1ec556e9674cb21791741e30a231d917e8cd1e37fe201fc7581e83f0408f865d50cc0cc9a0d13124ca867e3b70f7b0f39b73de0c750eaaa30fa77a3a6b630f57a1a9f12274aaf489223119590b9dfbca9047f981cd6cbb122106e0d771691491780b8e84513e6ead2ef08a09c0990c055dc61b5865ceb38d32c97248010658634a5d3f4fd500221957414aee4e8d842c5a8a9dc0dca9368cab4b891c88df2d75af1bae156e94914621afc646e3cfd6a82ce20b7ba080b698f129afc577382620fcc345559c604ecc1498a6c44ef486113b88eee94ff9b62aaac050b36ac5b4fde9e08f80dcf7ad03a69846e6ee7014f00ccbb29a3958b7ba4ecc1696c1ec5c74ccfff747aedf5e956e2b311f20e705cd2746e0ff0b322de033995bc01b30af2e3c7e16ea806bb5e9c1cdffeda4addae101771488242a9b052fa4aadfe5171f72aef7691a00315589580d0dd2e858f70421c8a7d6228681c1699a8e36ae241fea9ad858269cd4bffc7744d55f4a6cfc8f1ed8256964fa747d97d77074d4dc331e96ad8ec0b4aa57abc61200e3695e5dea4f1207a289dae5c220e7fc0b07a3fb6718f35758f646bf2fcb00fb1fb3a19dbe3902e921ffcc4817ab488b79a51fcb5c98265c5fc02e48e0dc7e3e64c7db2e736327245845333de68d7599313ba44577aad30855a0c5977791063543046fa29263fb63c5e89419d74dfbbfd239d5dc30b775d5db16912d4c8f159a4af7d37ca3a1fc01eb58837e32634769dd3e290220447b6f40a0357ce80d48b7dd10ab2f1fb3b589492375ddb073e521a2a71259cee8701f36cc1708d2b20c3889fd5b3c7eff94846fc7efaaa9e66171d30326458509e0701e08ba438503dc76d20c68274a0c257dbfb45577b73128c4dee0e957fb2f490c76d493df878935fd35cc6661f96ff4cfc9774bd740fab7a536bebff389cf164ed1be381f42a8a55625cbba9f15716f050d9f472bdf99aa9d6b0b3ebac97fd4a1af242b955ee8a70c3278f91116d710b80c8abbe7fb946e17fc34ca1032a800bf6fb4c2c424cfb779ec52867a7607665336c82ebe2d816eaf6aea06964dee26f2a3dea24e4cd45339a3f7072ddb68fc512ce61400ddd748be279958bf73a8b3e5ec005e762f8d06c609c4dfcd9f923f4c4f531042950c78eeb2a154ee6c6e340398d83bee55604e998e4a94bc3b099bbf2e9d8805dba76b9bacb07fc560a533d64ab4ef1956d4a74e9206ba8c4645be044b340fcd2a2ebd5f76352db718f41fc87224cc899c808d3fb577b5b049cd02b615a4fbfaf908da94afbe0345fdc164649fb678b1aeca7cf7791c1acaae06066f2e6ef131d0cc4ff599a984e48793be3523d85813f63903005d624e3deb873579e60073e449ff89dcf48627353ef169b4bf090f331b42abe4c2f8f47caad71c80ecf7f894bd0914c520d26e9a8c51136dc1d6f291767dc41f5a3e7219346c37fea3b3f3c5b28f7d48af2ccf97083937baccb0a6b1a041080a73b15b9640ccf44d9ad2e9f9b2e93fd21fe3debd3b203336402490162142d440bfe01576386fee1832b36fa346acb8fbc3a5f07f8d9cb26abca1432161f9b55c9f49343b83cb81c91c03e944d64085106ac1b950880521c2a2c53485fabc17b8825b4c48cb666bfab712fb60fa5b30ca45e5adc2a2f00dd892fee896442097ef09a4a24077d6e907848efa8a2b4328841580225876f3d306fdccd3e4e6bda25aa5a3ea950183a58a14c67b22d19ef8d83a0fcb846c771df8439d9ae44af9c395e6d799c900f30decd857f1cd85d50e3124031dd91c004b07151f8236f94b883b3dce245eeebbaeca5fc202dcd07bf51974088dc8de05f6cde2913e3f6a494c852ae60eb93323f48d1b8302e319fa3a1baa6e8fc64b2e8db72cb1337026799d3e3ebddd7a66f27bbf8c198672fad6499227c3b863ebb08c4d63ede67bb642f6b0d9b40525027d90dc2204120ecba6a0278ca68c2428527f0b28a5ebb4823bf5778c771a2b9e4bf06d9dc12a1934d1ca3639657f89997ae28adb8797bd22caf60cd77570f082d406ee2fdf2314aefe4468ceaf0706a7903326fce19ee8008694eb4b1dce3668bc40c888ccb1a392729b19acd1b71deb490d6a1d7255d7b3e53e9d47fe764fa85b3a9709084a28a7d54ad3648c0451dcfbcdeebd88510ea56d7f8d6e861dceda3760d451105126279cb020861c7324e52ee10f4db214ff6d85c4561f53091666d7300ad1836b2eca4f778b72218819ac5903665e7e162c74d4a961eac3dcc8e6c134a50abffc94d8540d7ec939b7a28b10916e505ad90843d08b4b51770d48c27beb2d8d548a1b0a50fe64ebb3964219ec1791d3c6d23562e0067cd3eb1f38926dd02ea9bf9803b8aca6e86a2d889ac122026b494096164aa99760e27e96fadf2d6003107faaf16bedb373b50612ff2978ee753156993d7383bab22dcfe911f121375522033766ac9a80916cebd9f1ab47382a2c87500d6614eccae65daa1080efa28f8ee8c1c5f319bacbf23d2214df237c048b416f28aab3aa25dc863e78f6ff5e3c97aeeeb1c801ae7dd578afcb3d74269eee132af1aa068f5d40f83aaddbc6b6c6dcb3f29ca3d8f841873cc7a73edfc453b58b280afc83e8dbe422134197dc48c8de9691f90f161491a7d7fe1b6bcce1bd8207fad71aa2531c16725cb6c3b670bd1dd52159723f8ff88"))
                    (acl2::hex-string-to-bytes! "7ac9b2b81f25cb0c96971b4647943822bd86d0006fa26049cb9943a09dff78d287afc49831f5c5e8b225a25e0dadff44"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "92e47b82b728d639777d5d5843de2a5c364956cb4b21cabdced2529b10b3f4275f307fbc352866d7b094cfd7426ae801aac17ac72335c04adb8d791da69b3c4cf4640c9d01812c154bc534be492065a85ac33a2efc853f7a3e86264c781218f639c849ac81fbcffa3058b496bfc94a582e9abe873ddbf056df0582aeece0302e16a35f711e5df88f489576e60c2d3e1a0e8d2f5a50fd7be5a651b5cd6bcc2bfb733f757440714edb79c3a3369714eaf5e2e3b5309cb387307cb4e6315e9fb16638e2b43e50fd7ae6ac2c620fca46f5f192e5778f89c7bc3d47743e602b216c764dd54456b4ca23b8ca16dcf70c8a081606db9e027fffc416f1dae4e43c1a01339a604c44d6a0f25ab9ca3978c6aacb6d270d510ee62544c3602e0045f7521ba508caf4911d478fd36390a01a5e6ce5fddbc2b93d26ffe74a7eb8bc09c69160d57d5d66c4f086a12baeaf39643d99ffce0b43fe16a75652b531bf3eba109e78cae25fc765b926d06a84712ddecbf9c0e9a0d94721d2f1300c638daedf3c420c8c201942fcdff0c699cc1ca4107053d15580cabce70d9a2755fb683501b445c4241dd3fc12caa928c06ef9c5839570d91a955e87efc117df9ae199fba7df6f4799c3d5202ee11ca1e55150b9bea0198141c5dfcdc1573102e2fd68275925873ef3f966d4d98e10d7125f69a849fb1471d1fa93b9e635b03536371ce1a0ee57503ba6cf409a394c724e7f8bb9ad0c2950d21bc470e5c55c229d77ff5302a2a52678b4598ef948991ba2990014713d22c74d10b81d466efd711fa557a209a93a67b20a49e37e24bf7a14960ee21df5c67d48c1b7e692975acac4a96df93d6cdebcf190ea6ab2358636af5cfe4b3a9bdc1bce160bf350aa3cd3956b897e255158cd3e2e83481ce3b6f778d418764f992d48e4f7fb6d080e6b3799d3f35949c17241a0cc5ba84597166779e6a38ce45681ad944cce7c432baf9cd8caf2b33125f2c12052bbb0b3b76f2cb97be9b4813a9ff1e5fdcd478769d0ab5b36cfb466e3a1f4c5a8eb88f0ad89ee40793017afad51bde0f240f75f30950b2901d3836f65478397b7335984df2c5617cac1dcf5175db98b136fb350e8c7644a114349d411ac159e65db4bf5b7200b4554e42cdce08c8cf049a1a4e4f92e60c7506a2d1e519753e39b2a85cd9b2cb6d201e02feb70a6df177a1ff2a2cb92a5a7a48f58d48cd2e46d42693093f51affc7b7ff0caa307359068f32c412c6bbdbf2f7f9f17e2923724440e173d02cb08d2b6087eca186bba3c7712042bd5371abb130f121a49253a2717fd7770ba8b70906e3c1f659db4564620631962ad87bbdeb37bf10cf2e2a1efbe59663ccd1f61a6028f5535998da45dd38afe5e1bfa2b8a8f4abe586656f8eea5c65609c0f376942176b2a292342ee42eee05a6b6eae44f04ffdc9529fec9947cb8b9601ca409eb3986aa452e6450699100e1a4f4f670acc30fce3ba4467e81919512c856ada60bc933be95b47aa36e8857f5015c894b2437ff15c46bca9236830ff4bb057cd5764f02720ccb03b2539377b224643ffe45e24245b7e707af13cf7dd1a09f9e12cb39774722e88be7b930d4cb869276e472407dd09ebc6a1fa9e5632c218b9de87b1706a060737fce2ffef393b06dcaa839d10648d1427e8e79535486a65fd8b0f67645305e225bd0083c589d3a6ca7b41d128001081fc15602a7c653b792eb16ddc59759644bac97d2a7785a7d059a829c935fbbabbabf10c6413d25ddbb5184918381ae2b75ee702f9ee75e00cf1fbe28c209bb2499e44944be3d7f713f8884924ea8810c279e137fa10a1a5e2fa3a1522d22921b33bd69fe7d2bb31aa61c91807560749ee737dc334c6501fdbdc2e699432ec2e82ec6ed641c9c8342757a25cc963e1b5c1b4844faff9cb767eff9f87e9272e023d18d6e160b144281241ddbe73fb5c7bc5a4bd82748f5bd153cb5516299b35c46c26d4674c707835f48940e50c05c2c482abf2058a724658e3a917bf5c754b9d988031100ef0ac815396947a9c6859b77f32e97ff988d435b5ffae2a8b9db6b5488db486ad172837bad0a0242c0e6a7d449f6b99cabb4f8d5236e3febefabe3304b5276e31cf337dae9f2e8cc2de3c3d595c40fb04f1fd8b40b77ce3fb76ad4054605073a3ff167bb96b045969c0192047cf304caf14285409e941834cf8c903c1f6f93b0a8f1a5b629ceb34269c474e95620334d40b16133433fc83b8695ca5d2dfe86e6e668442319aabc5035a0feecb7c805d79be0fee4aff738e47afac4d1084e6b5d76e9f5aafce0cb307e3b9287184716ca9155d0dd2b46f4a02b30224014bbab84e8eb2cb639ed6a73a1d8d2610e49c0c69f8479954b9b85ff05da5522f9d2e1edaae04bdd673d048a5f1d30d2f03c9e3c34f14ba597816761a96dffe22e3088f84b6de578eb6c43f4f47d2085749c299be020d97685851378644f650608311b5538f88c115cd78334108d33c0f1e4ed762f97a5909c130493e50337173e711354f1f4a24f322b4cd811a1ed4dfea3445b81da327c41b7d554aa3de89e375d0ba804f6ca8ae1b42b51967fbf64f153d8ac754cf2a22cffe02b0243fbf369715eaef083574d022e99210d6b2cbe807314e9887478aaa437cf97aba9d3cf6b4a96da330efb1e19e700c6f4e8a7a3129d83b319f546ee5e0bee1b04f4228aa3355eca4b0e08667f7a0cc3db9cc2bb3e1f9f79c823cfe590e8e6358ca74aadb7edf0ffb71501cec94417a82fec108f4985e2ae11a47649036b2277c127cea10fc0f7daf5a899af7df1bbe4705ae367eb7cd7d064dbfc512bef3d42311a211d115afdfc852d229113ff53e93b6dee0d9a40f971505ea4c6510191256ae2d47a73185dc4f4372f1c01c98a56ef9982c8bc9ec221529121e1392ec8b1cd167b318b85b1d4b14b2198fdfcd99e0b623421e8d2bc30fa17d66a4ae34e47d62967eae12f749c4addaca03ee5ac2f2d5c84b1a5796240a55734a9ecbb8c8814a03cff1a82109c2f6322ba6b207caab3ecf6f0591f3ed0e2d3ef28b5041dd316ff6cf68e0a3054ed70165ff939e5614d980d569a20bc624112587f2a797cb94bab6a69b2344e46f38919f96fae511338f22a6cdf2a65552e96f315c37823bb2138d7b12dacb4f2e012802f44da6a01246ff09bae98c45a29d65175068cad59463c176566b01ed911a0fb6a0f5a7aeb9e75d27b7ef843c1c154f6256a2364041909bc1ee40ae67a36b8ee2225bb1e12e5b85284c6c57e2c7397539f64d8f25c184638162bf9a6469dcecae03e68be17b88bf3301c8cde1abf21ec5b9d19d2b1d2561e13b528c8ded567343cafdff57a85853705f6baeded026932e269728fb5c5ddd0700604d8a36bf69c9ff0ad3ebbbb28808edbdc2843fbe1d9bc605f83c86131e1b3951a6024c0117672b74d044be7da7233327d178ffa302341f84d497a42f6d466e8c5c9d3802c546d5a3be0bb8a3f8e8ba0fa26518bb8a23bb7d86b4bd9812af287b3a0401acbb8e85ff65e7b20bc38009defef193f0ff8d2256d43ad9385580ee8507dfafe54066c3f08c81b39d5013a9fdadc47729a79dda7987d200f67dde49128ad54b979df618defd80c341e6c820d27787cc4ecc786d4c67233bbad4f3c08a8c8fdba91b670b317e8deadfcfc7f8c724e29a46b44963393bd6916ca0e7db56d5b56a0300e12551d850315f5139b8437b7ceafba73642a6cd920bf52ae222dc2deefd8d948beca6d3709fa9b546e9ae45fec354ef6794fde300b03bba0b1e855ce29e2f218eea9936963da64315126f4a0f2e39d305e0a546c78827988a56b9293a54d1ed12a790cb69123c42cfbbf0525832929a38af06be909ba10b8852e15e9e285b413eaaf458922654fc9b6d982581d7478d2788881203f2925c2d9b63c1b520b5653a7b2ea6b881d7b7a2aecf8a312541f4fbf3930cc794e0d4a0b8a91b2847c7a12fa2a47d842cabbc0b09fd4b10256617db788f72858975b90a3119339ea27a3d45a7c0a49e2eba53ff038a2ec7de9f3df0a0e40de4da3753928f1dd3c03bb480a313e9e5ad8feac3a626dffcfe58d5af62f2faa58d8e10ec61591cc136195eb4726e86065957255627f9627c5c9ceeb658f73a0227ade7ca9e177ec6a744611fd93e01d04c8f5252ab047c788ab1d498fa350aeb5b40a3367252aaaabcc5a309010dc10b2b3f590059d095b6309f6d9d7136d329d14d2e7805900a351ca30ebc11622e7fa16d22a85ae41600f56fa30fe582fe8530df6fd87e5c052f6fe6897f32c6b0f354239492d5c8651c81285c347d9ff1c421fb4e9e053b035f5f97fb998cc0af31ec689905f305ecd27dd28256621716b14babb311c4b76c8e92bc4e5346d893ff1245184bdc2212ec9176fcbbbc72d41357f6867ae057a984b06069f58996c4266ec0faaab7a73e30641bb2abb23692b1ece5af2184a32009628834c77c83ab6ea7e4256bac1dda689fc98c706cedba54b37e174ed6243ae5503e97104826ae9e65b682d5e50b4d5ae9887251131eb3c453cba449fe615fdf21897a1a140736805240f93287f9ffcfeb614c95cf52e5c975d43f5925e3f8c17d72c52b09afacda148973c8bd6f588e4d5b0285c710b82bc1a0d1c873306b510eff608c4fa576755399d6e1adf6c061b67f7c6924e094b1de3740dc0fd22258793a33850d71fe19b8790cc37f159f0eced4a3890d4ae12575a176de2251839d692b275d6b9caa8dea30fbb65c555d5e3e625d2cabd7074d2de65915d333f8ea84d9225e92ade49b92c68d046c3d6219ae31939b260e6ef1af0fd3cb4b02e805724c0dceaa2c13095844a8f3d6551607c54a2e1fa5787edba324cb46b145bbb7dc8cd1f6daa29ea0e3b136cdc325e2ef2cb414fe509ba0d2af2b28d02e85d8c86aa2541e4b96ef28dc151ed85a362c9b8b772f8050013a9d7e2ed4afea49a900c395808e25906d69ea78485dc40f5dd1899c65efcd32b605b634b62b0c199527906d3c8d8f18802bb6e39ca32838de5ace29a091eeb57ad20046b68ebd7d040478326f64b1b79035e85d02f4af3a16fe0329c7b0dc11f0cc036bbc1e24fe5e6024ed035cb4f4055027ef6296809c6c84d0a1c292bfd17eba68abb0256a230c033ba6dfbbc05848441ba0a4d2c5e64ab6dc9b797fbe7a8e66d60e7ae9b67574b56663ebc9fc6abfd96d78b0f0a3a1175ad01ed4c101968796f69d03b3317bff3e4d2e0fd091b2d3d6c57f6e3844543a395333bb9fb1acd113705f31ca02bd1565970e3316a7b8ec05cabec7f2d0871d1b0ee11660f645d4a5eb5ded7853c31ff3ed0c3fe46405732c253ca1656ec227c8a830cd4df7e8f7130f9d2003491683b1f62e1b82f320ce147e01e6da478ec007ca28c8aedd1da5278529b0b40e3dab645b7eb0387a9167857b167c99837250a11cd2efcbcf6d43a7f50987d3cac0fdd7c304e44e22e1a547c7824dea49f73f5f45352bebf65e3a5c06438ba8a428e3d1d7c7a00b1297ef21084d3f15c1a3f607ee5962577a7af20e61e1a67ac5fbc4193fcfcdb9733bafdce7803c611501db61be477d588249d385aeeabf9355c40878a224519e6c554c400b222b186a9657375e066534365246889467bd397b0816b6a8cf1602266e2e151d3bf380ebecd34f759e0033bf1db1e1bb5f8ab55f51589bfad7022fc80dbd23c4d47ee1bd8ab572ac18389f45327d56ccda0d4f2a2635ee4044ff6c7bbb6132a9f3b3850bf86668ab7bf4beef1a13b06fe12e4e04101579211ef40067ff820973a37ab3fbbf4ff919cf1cdae65cd9b64b7ad5fc14966002466ec834b3618871cbe9bbb2b566d7ed9f1d97395280393209b0ea9f83407f52573c65508869db3259cf5ecb6275650be46c4f235a0e9aea508855807896d831fb8eb29c0da0e314c45c4c462a74631c91c42be2268f64a8c90cfee71c21d467da6e975f6b8d08dfb5986b6b34ec38e84d2d13bddf8bfe321c43ce1b9768dbbb9de3fce3c9b9b35ad38ca481a9654c37c1b0effbad88d250f035ab6610f356f6ecf5c96e50aa4a4ec73fe0d2b06507d19454a7e5b71047ccedeb7e0eaf4babe06eade64927ea378338b905a47276d17e6f2224c326aac56dbc6c305c9ef74441cc123cbc7801e0c1cf03fc40171f1205fa204cdd21dfb4fa05c0a798027f87e1ae2db10e6875afd03424aeeec6e9322fdc58087ad1357d3c6ff0cd0a066cde88ba042e8921d00a770e2402f6a75a99c4bed241e35e3686fba30389facf45b0fd9ba29c73cb995b7f050cdf9fe3fcaf8c4795dd61ef6a38fe4901c2af88926509f70ce1158d5909b1c2e32f32a257dd4e20a9c63aac58fb0f08f3627257dc6574774ce7b790977cd3edb1d9f70bf189d4f016e2b6be8838f488e0508b64a91706c26894a1fcdb1cad2e7fc53647541de460e41b1f4d82557672d07442d884e586816bd21af8ae51092b54db377d08ff47181c57bb80cfbc5d6ecbb6dedbf236be5b5a886bac9683ad8f380341d69399c5bfc680d480f6d37123dafea9947d7374f93f9edd411d1a620033259e7a7f5125fde87db0757f047a3a0434e92cac02710221fbb6dfcb7ef0264e2994905491317c8c369786f3b63f5a5240e7cb074c7d1f09893ead27171fab90f76c4198c0afe1b993d3dbd850892cca0aeb9717b95ff8fe2fbfd4b94eb73402776eb4d54568d7787cab9db93c2058777b3dbb63b2e13408d57f7c09897f6caa92e112928a2404d0d966f41be1cc855e79245d749dc97b2f63c7cba03140f8922e0f8258f6aa10c9924d0788a7555b89edeaa6d0e3e0777cc2dfec05afc283c2343e2daef1c4a46b8fa8ff87f68284ffc0d5bd01edeecef6d80f05c069cdb3408db73a1ce33868bfb9bf03a1361ca5c8d82eca8f22bb28a68e9f3f5ac92930161fc2d22a34cf03ccf6d93c5ccfeb69ed1f3cf700e90e2e92cdd6c15a369da34cd4a61d614d511e9a2f3fd2432ab775757c496adeacb187c47bfeb44818f3b4e893ad4248948660324d3aecdbad487468d69b4bb0d26adc44815d3fadd6f10af4431d6a687236729b3610b2aec1660cce177bc3ffc302d518c83b8b0fda604e77ce3fbc475028fbddd4094d43cfd8e5b02631fa52e641194a37e2bff98b07277a1b5b616f9425a4010d6fe5223d67a815eb051731a5ee51e2b59904f051b03e0412b98068a8f5150ecb88a1d474dbfd064e3f422467d324e62ca7827e94213a1d9ba042b8f260f50cf59eaae0ddfc8b8a3726a5b495b63b608355602abc23c5cd9e5acfe9dc2eec0473f728cedf91fba98e03656895603d57a2ebc77c317e841d032d4e83cc26a4fd2ddb4227e164f46b3b1eadc9d099d3d471b1defd5cfa2114b262e70c2c11c2ff0a9420714d3b1fd926bfa35eae28a5ded7eece34a35f44bb9440c40e63b7f112f569eb378adf24fa382e6faf98d75a7dbfecbeaa6f7494188f110a10d9e0d73fd3a1cbfe0018aad757b23a44f46c9bb90ef1d7e52902ba368c1be49bf6541c8d8d85e75017ad59766c952ca62bb00f9bad46406f37d15d1a74a8783622da69050b5d68b43768dbe1601d6c5ca66b5f2a843322d003a926ce2dfeee69c53c66ad357175b3c80fd520199323eecc65326f7bffa58ef352bd59a1656fbc4740ba8ff30837d4326bf239813f78e5046531707bf4f0337193c9c3b62723b810af0db32e6bd69fdd44d7367f6894c7b83504376a582c47e436f314568b31e48850091fcce3799c32622f1efa579e1f4f9b52cc32748050b401e25568d8c1c818ed54ddb7b64854978ca0fadc55bf723160c335b99015d848bd3e8ed457fded264c96f5d52c6c4e758cd2d10ca81f4ac54c007ce9f413660a32b1c582500bda1c36786d108f0d1235286fab7298e52e7526b1a9a8e7a1836127e1045a0a13176073a14f4f2552293d291c91f66f3aa7a9b87ec2e0b2e7b5b565765243cf3e2d5ccbbba37a38641af75683898073453cc7c80dd1789e0e876fb519b828e5b3a376066b6f780e6b6dc9794c9ef8ee2f75381f09dd58f13e7ef5dd723cee959b0118eb5bbd33ef8eab1d581fd546261fda648518ef68f39e24b0a528e97e271ed0a94433949225f55b5abda23852643ef4b3c1e20e90ee570008d7e4c4c0e3639791d79afd7df7c905e86e80ce04efb34a8e9d1042b828909a66394dd79c1193de376d39fedcac32312f335f9f4511a7645bdf4a93c8ea727b30950c29fbf8d52e05cdb8a2978d576897d6e8b4fb1042cb8350370aa87c252ad0b1c8c2c3e1520d62c1beee97e26131881dc2141035ab066fb880ae4026d02fac552aabde143206b60ff3ecc2f377156ad295c76ac80b9d86dfcd8878309c132e4bd9eb32d99c9e6a8001a30107c29ba5bb0bd6f31189192f8cc59e39e1b05dbc5a7e8aa978b85fe4dc8d1b3d676a4ba1a1ca4891574e01d48910fa69833595c049c1912a38afd08deb425d5c5c96adc64b7a252ee482dddc9c57a4f25c5244a1e149079c63a979cd74c570482c6e4a7d83c476b1fe4b36af8e42a2bab530f1c7eabff4e93fcde30d41c15ce71f1ba92c0f504202eed492b52678feabd5f6249a2cca60f8154495b318dc2a237fc5523db8c59109fc437c562ea2c0ac06cd65826e1354b12c361729c2bee1e29293ce1646cb6bae6fc179fa6c67eea48edd3de77a42d7fccecaf2d50710890ea2aea421810cd9d991751309aa70927db64d333116fef2c14128ad21ae7a2b8ef53e5cc4312de272a888d1ebae64b31ec646d0c80beface8ce79d07481a3cd4ad36a7f691dd98f61fd213b0840ec5a6f06ef9a1420be0d59bde5e43546347a2a865a94770eb6e78ed869dac82ab08e58bd711b7e1f131dfd569370cec6d4371655372356cc6cf2e180883594335757c135f3db6cce95e5bae1acfca6c92c0ef68729b46d1fa4b8433fa5d5d2228abd92b49e6147f7ed6c79a557102517e406b26557d026cf06429a5be840ecc0f0c9b38399357860c3ba23ebbd35b377a3273237eafee8a33997d01d7a0048d532820cea0ddf65d2bed89efb05f2b8b2117a5f4509c71c64e6bfcbcd839d5029f7f1863022e7781486cdd41d58d09c90d7061fd6ddb228dab4c375b32a9d6a83e81c079fffd618f0c4a1ab99de56c7d99dd45b338e4775cd4caab417988a00acac0d787dcd34ac2c205d0db6be0ca2550aeafca5430e6b05557c7240f5eb796decccc213b3879e914108e1bd217a93ccc752e872249072497841967768493d8c505fcba9da5955e7d0aa0ff62e2903057c1409e25a54f3635d020804192bb4d17fd07d876a3139f6744c7ea90ea05bc9d036453f0de25419abc7b02f311946062227bff48a98b285d6f4181ee86701e06bc286cf64c3ff340a967640d89b5bf1e06a783a6215285e9dadc49de5075f12f6bfb3dc887e5a7c2479427e353a5b5c62a3929f1b300347727c6a6d2d07db271936b4aca19bcfb89e77fc32717dae0d50d9f46cdd71ab09fd6bf3d7e7efdfe67b2e7187d6a3c5b223988ca4b3edbccc511586e8cf61896ce58145233c48d218c5afd1b74f1da43059e5da1c2410ed630d5e305d1a870337d3c6499953981c859ec2bf833591d3fbe865cee2ec71aa982dbcda11e56b82346e43e780556a25f5af6d2032b087bcdb7921513d3295593559376c4d59966f295fff082269435430a0badd0de788c697b41f49771e06640038d8ce599e8f30df4d0292c5b8ad13fd18b1de5c031b59f94069a0d6da4b31c273a6e97ddbde2f1ee1556fb2692fca26bf590b672bc9ddc7c6073ca38bec2a42b1a9a301390cd2adc3ff8531a552df713eba4c758819bacaedff07255c71b54a9beae52cdfa083569a08be89949cbba73ddc8acf429359ca5e5be7a673633ca0d9709848f522a9df0e7900045d97c69063c8abbe80bf208fc023c66cf84a260d1244a29ba51bab11c95935e829cb68f3598599886a2678e0bc5d8bc014acdb59ee6ced6e1512f3b470b0447e1fbf0a99c71b3a75294fca86fa131eeaee13c34c17f25f086d8fe890ae87150e68fb124b539378cfe5c27c4a214c6732b00646ae32dde2b3d61af26c28e590e02b07eacfa8f43adcc508f8f2d9114de07f0ecc46ea365db5783c791289d29d3b916cf8ed6c4667e6f2a8a446cc69bb3b86b2bddde404b7577a7ce54877562469c2b1a66f4891e933942aea10a05c968cea7a813bcb4fdae104639fd79a7a0db0e55e5b1be3dea70bd2173116fdf2c71bd6ee80284520ad9b40780d879c45405922fb7ebbe63489b8d4f6e0218c3436307097226cfa7e18cc5443981f22c0a5aba2117915f89c7781c34f61f9f429cb13e0fcd0ce947103be684ca869d7f125f08d27b3f2c21d59adc7ab1b66ded96f0b4fa5f018b80156b7a51ca62b60e2a66e0bc69419ebbf178507907630f24d0862e51bec101037f900323af82e689b116f427584541c8a9a51ac89da1ed78c7f5ec9e52a7f441ad5e1382e083a95224f395dbceb69fd87c859dd1baffa369ebd48248b142cfcb2d0a1df813d178f8147148223312135bd8ae4c3d05d8ab68ca9681d9610e1df08d38a4ebafe8cffc0e04ada33be446d35c93b852f4c97005d2a05dd8d88649523c7e2ae61b5628de504228c73fb9a4cf56ba74dc0e5e8bb192a688bf8e37ec70c92f226fec961e0fa9ae3c06c4eea7b07596e400c8e4790c990da506089f71dc91e02df1aa251a10689d6a0a420fea1727cd8889ab07a277444a1544be2973e64515a013ca8fa7712bb4351a5da9c673fbb2f4df2b44a27222d05be445626d3e41541b3ed43240b251772fef7e28d7575fdb9b69d007f39db3550b6d04d6b835251011db4db96054785aacd20a299721c2cc0ec22fee104a843c866792f2ff7b5ede06819d8190c80ed34e6aa8087073d2f8a17091537319040b7f4c6643ad615d027bca4a275c8d93758721de621dbdfee5f5ddbac4b679c1263883f5454c97be41d40dea172e2594c5e99ff3b2d970a5189e91f2de34fd3a2154b6d674578ed100c1d4e3dd70209128e82dae774e2c6404fe15dffc8903dd81464d706d48a763c3a7a7ddf6ecfef7230"))
                    (acl2::hex-string-to-bytes! "4a86527c19095982e491c56f5fad57593d827683ae08f0ce40b607a49ce25222abb4b96762596e4809574222307e2e47"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "c55c7d5fedf6c5f93126dd90b6f19720f56f791590d42903c78a778063b93ed5174502c6b76b0d2b75658ce813e867c3"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "13189ce20ec85fa2c86f0c51fa9de0d0678e6ab13aba4ef0856b37b2daff99b5530b258901b6297c68d542513082c115"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "43e98dd681cedf885992d37e500247fef6beadebc6d90eb05c3cae3d808daee1a483087606b5b9774903ed672ee9645b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4cc9b319af23213a45f5b8ec966905e20aa06248500cc2bc736b171c9fdcad3d67493f930753912d82125e8bc188e72b"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "63c337417fa6fe40cb3c6ae8b9d9092f4f3e7f4dde7e583d94f540383eebb0c5a80daf3b4690861c5b0e5b0d29fc448d"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "025dc380479269ae1739ed9631f7606fd2927f71893ad74ae864d6af7d14323be5fda67b2c456fff98387239b8eb3a13"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "810231a63c06bce7f411f1b8ef106c364564a4b90fb9b9b275c0749ad3c83983afe2044d1844d8ef76c03a1acef7c6b3"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b3b87315402171643f4f2826ccbfa633607fa9eecdeb224395b69ddac04d01b1159034fbcde1e0a7fb3c38155ecb6062"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "3d71f08e52812bd83b34694c14eefa3ae29bf4084e608c7832d23edd52d33972ec3164f40477f4c45ae52d147268ec8c"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "6a76b88ce50bab0682ee7cb670821ea8f608e7d70801e7785d26f9823455767914c5c87f0054a576bdd69938a6e6d9ec"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "5377572306ef61982330034f0a28fdde93da57c53cd1365126cd496847bc05694fba3d508b55c97db476a75876ff6474"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "d3e8ef0c120279753ba4211888c2b079f5913a1d9828500cfc68f29c9b182a8d38608cc7dae41448198ab428231c24bd"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "6486d90890cc9b399c800127f495bf67f692511b6f26c08c53aac308d3d3583e7184bec5ca9d17905d5b9717d43fa0dc"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b0181daadc927c25b4a8b4f634aea80911d31181a379d3d4689f164e073fff0f268f518ad3e131122439edf25c53bcd9"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "cca52ac605ab74649fd6c6bc13da780e913bf742ca25fdf24b0092170ec4909514027d8dc1e3036105219fd5a2f260b8"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4da010129dd22d2daf6da3a9c760e3ba8a68ccbed9b59c8525b84ad72ec04be130042794b8c03a573ec41804c9de3eca"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "3e82f9f2941e2e8edbd0942f8db71b923cfd2c62cc80f50e7064385616c954eb05bb71a0e52d5bc73154d24ee01458a2"))

(acl2::assert-equal (sha-384-bytes (acl2::hex-string-to-bytes! "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